#include "opencv2/core/core.hpp"
#include "opencv2/core/internal.hpp"

#include "haarfeatures.h"
#include "cascadeclassifier.h"
#include <fstream>

using namespace std;
using namespace cv;

CvHaarFeatureParams::CvHaarFeatureParams() : mode(BASIC)
{
    name = HFP_NAME;
}

CvHaarFeatureParams::CvHaarFeatureParams( int _mode ) : mode( _mode )
{
    name = HFP_NAME;
}

void CvHaarFeatureParams::init( const CvFeatureParams& fp )
{
    CvFeatureParams::init( fp );
    mode = ((const CvHaarFeatureParams&)fp).mode;
}

void CvHaarFeatureParams::write( FileStorage &fs ) const
{
    CvFeatureParams::write( fs );
    string modeStr = mode == BASIC ? CC_MODE_BASIC :
                     mode == CORE ? CC_MODE_CORE :
                     mode == ALL ? CC_MODE_ALL : string();
    CV_Assert( !modeStr.empty() );
    fs << CC_MODE << modeStr;
}

bool CvHaarFeatureParams::read( const FileNode &node )
{
    if( !CvFeatureParams::read( node ) )
        return false;

    FileNode rnode = node[CC_MODE];
    if( !rnode.isString() )
        return false;
    string modeStr;
    rnode >> modeStr;
    mode = !modeStr.compare( CC_MODE_BASIC ) ? BASIC :
           !modeStr.compare( CC_MODE_CORE ) ? CORE :
           !modeStr.compare( CC_MODE_ALL ) ? ALL : -1;
    return (mode >= 0);
}

void CvHaarFeatureParams::printDefaults() const
{
    CvFeatureParams::printDefaults();
    cout << "  [-mode <" CC_MODE_BASIC << "(default) | "
            << CC_MODE_CORE <<" | " << CC_MODE_ALL << endl;
}

void CvHaarFeatureParams::printAttrs() const
{
    CvFeatureParams::printAttrs();
    string mode_str = mode == BASIC ? CC_MODE_BASIC :
                       mode == CORE ? CC_MODE_CORE :
                       mode == ALL ? CC_MODE_ALL : 0;
    cout << "mode: " <<  mode_str << endl;
}

bool CvHaarFeatureParams::scanAttr( const string prmName, const string val)
{
    if ( !CvFeatureParams::scanAttr( prmName, val ) )
    {
        if( !prmName.compare("-mode") )
        {
            mode = !val.compare( CC_MODE_CORE ) ? CORE :
                   !val.compare( CC_MODE_ALL ) ? ALL :
                   !val.compare( CC_MODE_BASIC ) ? BASIC : -1;
            if (mode == -1)
                return false;
        }
        return false;
    }
    return true;
}

//--------------------- HaarFeatureEvaluator ----------------

void CvHaarEvaluator::regenerateFeatures(){
	features.clear();
	generateSteady = false;
	generateFeatures();
}

void CvHaarEvaluator::initFeatureDiff(){
	for (int i = 2; i <= 4; i++){
		featuresDiff["haar_x" + std::to_string(i)] = vector<std::pair<Feature, double>>();
		featuresDiff["haar_y" + std::to_string(i)] = vector<std::pair<Feature, double>>();

		featuresDiff["tilt_haar_x" + std::to_string(i)] = vector<std::pair<Feature, double>>();
		featuresDiff["tilt_haar_y" + std::to_string(i)] = vector<std::pair<Feature, double>>();
	}
	featuresDiff["x2_y2"] = vector<std::pair<Feature, double>>();
	featuresDiff["x3_y3"] = vector<std::pair<Feature, double>>();
}

void CvHaarEvaluator::init(const CvFeatureParams *_featureParams,
                           int _maxSampleCount, Size _winSize )
{
    CV_Assert(_maxSampleCount > 0);
    int cols = (_winSize.width + 1) * (_winSize.height + 1);
    sum.create((int)_maxSampleCount, cols, CV_32SC1);
    tilted.create((int)_maxSampleCount, cols, CV_32SC1);
    normfactor.create(1, (int)_maxSampleCount, CV_32FC1);
    CvFeatureEvaluator::init( _featureParams, _maxSampleCount, _winSize );
	//numFeatures = 4798;
	chooseSteadyFeatures();
}

void CvHaarEvaluator::setImage(const Mat& img, uchar clsLabel, int idx)
{
    CV_DbgAssert( !sum.empty() && !tilted.empty() && !normfactor.empty() );
    CvFeatureEvaluator::setImage( img, clsLabel, idx);
    Mat innSum(winSize.height + 1, winSize.width + 1, sum.type(), sum.ptr<int>((int)idx));
    Mat innTilted(winSize.height + 1, winSize.width + 1, tilted.type(), tilted.ptr<int>((int)idx));
    Mat innSqSum;
    integral(img, innSum, innSqSum, innTilted);
    normfactor.ptr<float>(0)[idx] = calcNormFactor( innSum, innSqSum );
}

void CvHaarEvaluator::writeFeatures( FileStorage &fs, const Mat& featureMap ) const
{
    _writeFeatures( features, fs, featureMap );
}

void CvHaarEvaluator::writeFeature(FileStorage &fs, int fi) const
{
    CV_DbgAssert( fi < (int)features.size() );
    features[fi].write(fs);
}

void CvHaarEvaluator::readSteadyFeatures(){
	std::ifstream ifs("steadyFeatures.txt");
	while (!ifs.eof())

	{
		char fType[15];
		int x[3];
		int y[3];
		int width[3];
		int height[3];
		float weight[3];
		double val;

		std::string line;
		std::getline(ifs, line);

		sscanf(line.c_str(), "featureType: %s val = %lf; reats and weights: %d %d %d %d %f; %d %d %d %d %f; %d %d %d %d %f; "
			, fType, &val
			, &(x[0]), &(y[0]), &(width[0]), &(height[0]), &(weight[0])
			, &(x[1]), &(y[1]), &(width[1]), &(height[1]), &(weight[1])
			, &(x[2]), &(y[2]), &(width[2]), &(height[2]), &(weight[2]));

		std::string featureType = std::string(fType);
		featureType = featureType.substr(0, featureType.length() - 1);

		printf("Read line: featureType: %s; val = %lf; reats and weights: %d %d %d %d %f; %d %d %d %d %f; %d %d %d %d %f;\n"
			, featureType.c_str(), val
			, x[0], y[0], width[0], height[0], weight[0]
			, x[1], y[1], width[1], height[1], weight[1]
			, x[2], y[2], width[2], height[2], weight[2]);

		Feature fea = Feature(winSize.width + 1, false,
			x[0], y[0], width[0], height[0], weight[0],
			x[1], y[1], width[1], height[1], weight[1],
			x[2], y[2], width[2], height[2], weight[2]);

		std::pair<Feature, double> p;
		p.first = fea;
		p.second = val;
		/*printf("%d - size of vector\n", featuresDiff[featureType].size());*/
		featuresDiff[featureType].push_back(p);
	}
}

void CvHaarEvaluator::chooseSteadyFeatures()
{
	std::ofstream ofs;
	if (generateSteady){
		int imgNumber = 10;
		ofs = std::ofstream("steadyFeatures.txt");

		std::vector<cv::Mat> testImages;
		std::vector<cv::Mat> testSum;
		std::vector<cv::Mat> testSumTilted;

		std::vector<float> values;

		std::string path = "C:\\training\\testImgs\\";
			//"C:\\Users\\Nik\\Desktop\\training\\testImgs\\";
			//"";

		for (int i = 0; i < imgNumber; i++){
			printf("%d image start loading\n", i);
			cv::Mat img;
			img = cv::imread(path + std::to_string(i) + ".bmp");
			cv::Mat imgRes;
			cv::resize(img, imgRes, cv::Size(1911, 486));
			img = imgRes;
			printf("img size = (%d, %d);\n", img.cols, img.rows);
			testImages.push_back(img);
			cv::Mat sum, sqSum, tiltSum;
			/*sum = cv::Mat::zeros(img.rows + 1, img.cols + 1, CV_32SC1);
			sqSum = cv::Mat::zeros(img.rows + 1, img.cols + 1, CV_32FC1);
			tiltSum = cv::Mat::zeros(img.rows + 1, img.cols + 1, CV_32SC1);*/
			cv::integral(img, sum, sqSum, tiltSum);
			cv::Mat sumCorr = cv::Mat::zeros(img.rows + 1, img.cols + 1, CV_32SC1);
			cv::Mat tiltCorr = cv::Mat::zeros(img.rows + 1, img.cols + 1, CV_32SC1);
			
			for (int y = 0; y < sum.rows; y++){
				for (int x = 0; x < sum.cols; x++){
					sumCorr.at<int>(y, x) = sum.at<int>(y, x * 3);
					tiltCorr.at<int>(y, x) = tiltSum.at<int>(y, x * 3);
				}
			}
			sum = sumCorr;
			tiltSum = tiltCorr;

			printf("sum size = (%d, %d); tiltSumSize = (%d, %d)\n", sum.cols, sum.rows, tiltSum.cols, tiltSum.rows);
			testSum.push_back(sum);
			testSumTilted.push_back(tiltSum);
			printf("%d image loaded\n", i);

			/*if (i == 0){
				std::ofstream sumF("sum.txt");
				std::ofstream sumT("tilt.txt");
				for (int y = 0; y < sum.rows; y++){
					for (int x = 0; x < sum.cols; x++){
						sumF << sum.at<int>(y, x) << " ";
						sumT << tiltSum.at<int>(y, x) << " ";
					}
					sumF << "\n";
					sumT << "\n";
				}
				sumF.close();
				sumT.close();
			}*/
		}

		for (int i = 0; i < imgNumber; i++){
			std::map <std::string, vector<std::pair<Feature, double>>>::iterator cur;
			for (cur = featuresDiff.begin(); cur != featuresDiff.end(); cur++){
				printf("start %s on img %d, feature to proc = %d\n", cur->first.c_str(), i, cur->second.size());
				for (int j = 0; j < cur->second.size(); j++){
					double var;
					if (i == 0 && j == 0){
						var = getVariance(cur->second[6].first, testSum[i]
							, testSumTilted[i], true);
					}
					else {
						var = getVariance(cur->second[j].first, testSum[i]
							, testSumTilted[i]);
					}
					cur->second[j].second += var / 10.0;
					printf("processed %d; 0.1*var = %f\n", j, var / 10.0);
				}
				printf("end %s on img\n", cur->first.c_str(), i);
			}
		}
	}
	else readSteadyFeatures();
	/*double valM = featuresDiff["haar_x2"][0].second;
	printf("%f - val\n", valM);*/

	std::vector<std::pair<Feature, double>> steadyFeatPatt;

	std::map <std::string, vector<std::pair<Feature, double>>>::iterator cur2;
	for (cur2 = featuresDiff.begin(); cur2 != featuresDiff.end(); cur2++){
		for (int j = 0; j < cur2->second.size(); j++){
			string featureName = cur2->first;
			float val = cur2->second[j].second;
			steadyFeatPatt.push_back(cur2->second[j]);

			printf("featureType: %s; val = %lf; reats and weights: ", featureName.c_str(), val);

			if (generateSteady){
				ofs << "featureType: " + featureName + "; val = " + std::to_string(val) + "; reats and weights: ";
			}
			for (int k = 0; k < CV_HAAR_FEATURE_MAX; k++){
				cv::Rect rect = cur2->second[j].first.rect[k].r;
				double weigh = cur2->second[j].first.rect[k].weight;
				printf("%d %d %d %d %f; ", rect.x, rect.y, rect.width, rect.height, weigh);
				if (generateSteady){
					ofs << rect.x << " " << rect.y << " " << rect.width << " " << rect.height << " " << weigh << "; ";
				}
			}

			/*if (cur2->second[j].second == 0.0){
			int n;
			scanf("%d", &n);
			}*/

			printf("\n");
			if (generateSteady){
				ofs << "\n";
			}
		}
	}

	printf("start sort: %d length\n", steadyFeatPatt.size());
	for (int i = 0; i < steadyFeatPatt.size() - 1; i++){
		if (!(i % 1000)) printf("sorted %d\n", i);
		double min = steadyFeatPatt[i].second;
		int indMin = i;

		for (int j = i + 1; j < steadyFeatPatt.size(); j++){
			if (steadyFeatPatt[j].second < min) {
				min = steadyFeatPatt[j].second;
				indMin = j;
			}
		}

		if (indMin != i){
			std::pair<Feature, double> temp = pair<Feature, double>(steadyFeatPatt[i].first, steadyFeatPatt[i].second);
			steadyFeatPatt[i] = pair<Feature, double>(steadyFeatPatt[indMin].first, steadyFeatPatt[indMin].second);
			steadyFeatPatt[indMin] = temp;
		}
	}

	
	int lastPattNumb = ((double) steadyFeatPatt.size()) * 0.45;
	printf("steady Feature patt size: %d\n", steadyFeatPatt.size());

	printf("end finding steady\n");

	printf("%d - features size before choosing steady\n", features.size());

	int numb = 0;
	std::vector<Feature> featuresTemp;
	for (int i = 0; i < features.size(); i++){
		
		if (!(i % 1000)) printf("processed %d of %d\n", i, features.size());
		
		bool matchesAllRects = false;

		for (int j = 0; j < lastPattNumb; j++){
			bool matches = true;
			for (int k = 0; k < CV_HAAR_FEATURE_MAX; k++){
				cv::Rect rectF = features[i].rect[k].r;
				double weightF = features[i].rect[k].weight;

				cv::Rect rectFPatt = steadyFeatPatt[j].first.rect[k].r;
				double weightFPatt = steadyFeatPatt[j].first.rect[k].weight;

				if (rectF.width != rectFPatt.width || rectF.height != rectFPatt.height
					|| std::abs(weightF - weightFPatt) > 0.00000001){
					matches = false;
					break;
				}
			}

			if (matches) {
				matchesAllRects = true;
				break;
			}
		}
		
		if (matchesAllRects){
			featuresTemp.push_back(features[i]);
			steadyFeatureMap[numb] = i;
			printf("map %d(numb) -> %d(i)\n", numb, i);
			numb++;
		}
	}
	features = featuresTemp;
	printf("%d - features size after choosing steady\n", features.size());
	featuresDiff.clear();
	numFeatures = (int)features.size();
	cv::waitKey(500);
}

double CvHaarEvaluator::getVariance(Feature f, cv::Mat& sum, cv::Mat& tilted, bool stop){
	int w = f.rect[0].r.width;
	int h = f.rect[0].r.height;

	float maxVal = 0.0;
	float minVal = 10000000000.0;

	cv::Mat featureVals = cv::Mat::zeros(sum.rows - winSize.height - 1, sum.cols - winSize.width - 1, CV_32F);
	for (int x = 1; x < sum.cols - winSize.width; x++){
		for (int y = 1; y < sum.rows - winSize.height; y++){
			
			cv::Mat sumRect = sum(cv::Rect(x, y, winSize.width + 1, winSize.height + 1));
			cv::Mat tiltRect = tilted(cv::Rect(x, y, winSize.width + 1, winSize.height + 1));
			
			float val = f.calcOnRect(sumRect, tiltRect);
			featureVals.at<float>(cv::Point(x - 1, y - 1)) = val;
			if (std::abs(val) > maxVal) maxVal = std::abs(val);
			if (std::abs(val) < minVal) minVal = std::abs(val);
		}
	}

	if (stop){
		cv::Mat savedFeatures = cv::Mat::zeros(featureVals.rows, featureVals.cols, CV_8U);
		for (int i = 0; i < featureVals.cols; i++){
			for (int j = 0; j < featureVals.rows; j++){
				savedFeatures.at<char>(cv::Point(i, j)) = 255.0 * (std::abs(featureVals.at<float>(cv::Point(i, j))) / (maxVal + minVal));
			}
		}
		cv::imwrite("C:\\training\\featureVals.bmp", savedFeatures);
		int zal; scanf("%d", zal);
	}

	long double diffSum = 0;
	int n = 0;



	for (int x = 0; x < featureVals.cols - 1; x++){
		for (int y = 0; y < featureVals.rows - 1; y++){
			diffSum += std::abs(featureVals.at<float>(cv::Point(x, y))
				- featureVals.at<float>(cv::Point(x + 1, y)));

			diffSum += std::abs(featureVals.at<float>(cv::Point(x, y))
				- featureVals.at<float>(cv::Point(x, y + 1)));
			n += 2;
		}
	}

	/*if (stop){
		std::ofstream sumF("sum.txt");
		for (int y = 0; y < sum.rows; y++){
			for (int x = 0; x < sum.cols; x++){
				sumF << sum.at<int>(y, x) << " ";
			}
			sumF << "\n";
		}
		sumF.close();

		std::ofstream featureValsF("featureVals.txt");
		for (int y = 0; y < featureVals.rows; y++){
			for (int x = 0; x < featureVals.cols; x++){
				featureValsF << (int)featureVals.at<float>(y, x) << " ";
			}
			featureValsF << "\n";
		}
		featureValsF.close();

		printf("STOPPED, var = %f\n", diffSum / ((double)n));
		printf("(%d, %d) - fSize; %d - area\n", w, h, w*h);
		int asd;
		scanf("%d", &asd);
	}*/

	int pixelNumber = f.tilted ? (w + h - 1)*(w + h - 1) - (w - 1)*(w - 1) - (w - 1) - (h - 1)*(h - 1) - (h - 1)
		: (w * h);

	double res = diffSum / ((double)n);
	return res / ((double)pixelNumber);
}

bool CvHaarEvaluator::featureVectorContains(std::vector<std::pair<Feature, double>> featuresVec, Feature feature){
	bool res = false;
	for each(auto &p in featuresVec){
		Feature f = p.first;
		bool matches = true;
		for (int k = 0; k < CV_HAAR_FEATURE_MAX; k++){
			cv::Rect rectF = f.rect[k].r;
			double weightF = f.rect[k].weight;

			cv::Rect rectFPatt = feature.rect[k].r;
			double weightFPatt = feature.rect[k].weight;

			if (rectF.width != rectFPatt.width || rectF.height != rectFPatt.height
				|| std::abs(weightF - weightFPatt) > 0.00000001){
				matches = false;
				break;
			}
		}

		if (matches) {
			res = true;
			break;
		}
	}
	return res;
}

void CvHaarEvaluator::generateFeatures()
{
    int mode = ((const CvHaarFeatureParams*)((CvFeatureParams*)featureParams))->mode;
    int offset = winSize.width + 1;
    for( int x = 0; x < winSize.width; x++ )
    {
        for( int y = 0; y < winSize.height; y++ )
        {
            for( int dx = 1; dx <= winSize.width; dx++ )
            {
                for( int dy = 1; dy <= winSize.height; dy++ )
                {
                    // haar_x2
                    if ( (x+dx*2 <= winSize.width) && (y+dy <= winSize.height) )
                    {
                        features.push_back( Feature( offset, false,
                            x,    y, dx*2, dy, -1,
                            x+dx, y, dx  , dy, +2 ) );

						if (x == 0 && y == 0 && generateSteady){
							featuresDiff["haar_x2"].push_back(std::pair<Feature, double>(
								Feature(offset, false,
								x, y, dx * 2, dy, -1,
								x + dx, y, dx, dy, +2)
								, 0));
						}
                    }
                    // haar_y2
                    if ( (x+dx <= winSize.width) && (y+dy*2 <= winSize.height) )
                    {
                        features.push_back( Feature( offset, false,
                            x,    y, dx, dy*2, -1,
                            x, y+dy, dx, dy,   +2 ) );

						if (x == 0 && y == 0 && generateSteady){
							featuresDiff["haar_y2"].push_back(std::pair<Feature, double>(
								Feature(offset, false,
								x, y, dx, dy * 2, -1,
								x, y + dy, dx, dy, +2)
								, 0));
						}
                    }
                    // haar_x3
                    if ( (x+dx*3 <= winSize.width) && (y+dy <= winSize.height) )
                    {
                        features.push_back( Feature( offset, false,
                            x,    y, dx*3, dy, -1,
                            x+dx, y, dx  , dy, +3 ) );

						if (x == 0 && y == 0 && generateSteady){
							featuresDiff["haar_x3"].push_back(std::pair<Feature, double>(
								Feature(offset, false,
								x, y, dx * 3, dy, -1,
								x + dx, y, dx, dy, +3)
								, 0));
						}
                    }
                    // haar_y3
                    if ( (x+dx <= winSize.width) && (y+dy*3 <= winSize.height) )
                    {
                        features.push_back( Feature( offset, false,
                            x, y,    dx, dy*3, -1,
                            x, y+dy, dx, dy,   +3 ) );

						if (x == 0 && y == 0 && generateSteady){
							featuresDiff["haar_y3"].push_back(std::pair<Feature, double>(
								Feature(offset, false,
								x, y, dx, dy * 3, -1,
								x, y + dy, dx, dy, +3)
								, 0));
						}
                    }
                    if( mode != CvHaarFeatureParams::BASIC )
                    {
                        // haar_x4
                        if ( (x+dx*4 <= winSize.width) && (y+dy <= winSize.height) )
                        {
                            features.push_back( Feature( offset, false,
                                x,    y, dx*4, dy, -1,
                                x+dx, y, dx*2, dy, +2 ) );

							if (x == 0 && y == 0 && generateSteady){
								featuresDiff["haar_x4"].push_back(std::pair<Feature, double>(
									Feature(offset, false,
									x, y, dx * 4, dy, -1,
									x + dx, y, dx * 2, dy, +2)
									, 0));
							}
                        }
                        // haar_y4
                        if ( (x+dx <= winSize.width ) && (y+dy*4 <= winSize.height) )
                        {
                            features.push_back( Feature( offset, false,
                                x, y,    dx, dy*4, -1,
                                x, y+dy, dx, dy*2, +2 ) );

							if (x == 0 && y == 0 && generateSteady){
								featuresDiff["haar_y4"].push_back(std::pair<Feature, double>(
									Feature(offset, false,
									x, y, dx, dy * 4, -1,
									x, y + dy, dx, dy * 2, +2)
									, 0));
							}
                        }
                    }
                    // x2_y2
                    if ( (x+dx*2 <= winSize.width) && (y+dy*2 <= winSize.height) )
                    {
                        features.push_back( Feature( offset, false,
                            x,    y,    dx*2, dy*2, -1,
                            x,    y,    dx,   dy,   +2,
                            x+dx, y+dy, dx,   dy,   +2 ) );

						if (x == 0 && y == 0 && generateSteady){
							featuresDiff["x2_y2"].push_back(std::pair<Feature, double>(
								Feature(offset, false,
								x, y, dx * 2, dy * 2, -1,
								x, y, dx, dy, +2,
								x + dx, y + dy, dx, dy, +2)
								, 0));
						}
                    }
                    if (mode != CvHaarFeatureParams::BASIC)
                    {
                        if ( (x+dx*3 <= winSize.width) && (y+dy*3 <= winSize.height) )
                        {
                            features.push_back( Feature( offset, false,
                                x   , y   , dx*3, dy*3, -1,
                                x+dx, y+dy, dx  , dy  , +9) );

							if (x == 0 && y == 0 && generateSteady){
								featuresDiff["x3_y3"].push_back(std::pair<Feature, double>(
									Feature(offset, false,
									x, y, dx * 3, dy * 3, -1,
									x + dx, y + dy, dx, dy, +9)
									, 0));
							}
                        }
                    }
                    if (mode == CvHaarFeatureParams::ALL)
                    {
						
                        // tilted haar_x2
                        if ( (x+2*dx <= winSize.width) && (y+2*dx+dy <= winSize.height) && (x-dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x, y, dx*2, dy, -1,
                                x, y, dx,   dy, +2 ) );

							Feature f = Feature(offset, true,
								x, y, dx * 2, dy, -1,
								x, y, dx, dy, +2);

							if (!featureVectorContains(featuresDiff["tilt_haar_x2"], f) && generateSteady){
								featuresDiff["tilt_haar_x2"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                        // tilted haar_y2
                        if ( (x+dx <= winSize.width) && (y+dx+2*dy <= winSize.height) && (x-2*dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x, y, dx, 2*dy, -1,
                                x, y, dx, dy,   +2 ) );

							Feature f = Feature(offset, true,
								x, y, dx, 2 * dy, -1,
								x, y, dx, dy, +2);

							if (!featureVectorContains(featuresDiff["tilt_haar_y2"], f) && generateSteady){
								featuresDiff["tilt_haar_y2"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                        // tilted haar_x3
                        if ( (x+3*dx <= winSize.width) && (y+3*dx+dy <= winSize.height) && (x-dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x,    y,    dx*3, dy, -1,
                                x+dx, y+dx, dx,   dy, +3 ) );

							Feature f = Feature(offset, true,
								x, y, dx * 3, dy, -1,
								x + dx, y + dx, dx, dy, +3);

							if (!featureVectorContains(featuresDiff["tilt_haar_x3"], f) && generateSteady){
								featuresDiff["tilt_haar_x3"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                        // tilted haar_y3
                        if ( (x+dx <= winSize.width) && (y+dx+3*dy <= winSize.height) && (x-3*dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x,    y,    dx, 3*dy, -1,
                                x-dy, y+dy, dx, dy,   +3 ) );

							Feature f = Feature(offset, true,
								x, y, dx, 3 * dy, -1,
								x - dy, y + dy, dx, dy, +3);

							if (!featureVectorContains(featuresDiff["tilt_haar_y3"], f) && generateSteady){
								featuresDiff["tilt_haar_y3"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                        // tilted haar_x4
                        if ( (x+4*dx <= winSize.width) && (y+4*dx+dy <= winSize.height) && (x-dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x,    y,    dx*4, dy, -1,
                                x+dx, y+dx, dx*2, dy, +2 ) );

							Feature f = Feature(offset, true,
								x, y, dx * 4, dy, -1,
								x + dx, y + dx, dx * 2, dy, +2);

							if (!featureVectorContains(featuresDiff["tilt_haar_x4"], f) && generateSteady){
								featuresDiff["tilt_haar_x4"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                        // tilted haar_y4
                        if ( (x+dx <= winSize.width) && (y+dx+4*dy <= winSize.height) && (x-4*dy>= 0) )
                        {
                            features.push_back( Feature( offset, true,
                                x,    y,    dx, 4*dy, -1,
                                x-dy, y+dy, dx, 2*dy, +2 ) );

							Feature f = Feature(offset, true,
								x, y, dx, 4 * dy, -1,
								x - dy, y + dy, dx, 2 * dy, +2);

							if (!featureVectorContains(featuresDiff["tilt_haar_y4"], f) && generateSteady){
								featuresDiff["tilt_haar_y4"].push_back(std::pair<Feature, double>(f, 0));
							}
                        }
                    }
                }
            }
        }
    }
    numFeatures = (int)features.size();
}

CvHaarEvaluator::Feature::Feature()
{
    tilted = false;
    rect[0].r = rect[1].r = rect[2].r = Rect(0,0,0,0);
    rect[0].weight = rect[1].weight = rect[2].weight = 0;
}

CvHaarEvaluator::Feature::Feature( int offset, bool _tilted,
                                          int x0, int y0, int w0, int h0, float wt0,
                                          int x1, int y1, int w1, int h1, float wt1,
                                          int x2, int y2, int w2, int h2, float wt2 )
{
    tilted = _tilted;

    rect[0].r.x = x0;
    rect[0].r.y = y0;
    rect[0].r.width  = w0;
    rect[0].r.height = h0;
    rect[0].weight   = wt0;

    rect[1].r.x = x1;
    rect[1].r.y = y1;
    rect[1].r.width  = w1;
    rect[1].r.height = h1;
    rect[1].weight   = wt1;

    rect[2].r.x = x2;
    rect[2].r.y = y2;
    rect[2].r.width  = w2;
    rect[2].r.height = h2;
    rect[2].weight   = wt2;

    if( !tilted )
    {
        for( int j = 0; j < CV_HAAR_FEATURE_MAX; j++ )
        {
            if( rect[j].weight == 0.0F )
                break;
            CV_SUM_OFFSETS( fastRect[j].p0, fastRect[j].p1, fastRect[j].p2, fastRect[j].p3, rect[j].r, offset )
        }
    }
    else
    {
        for( int j = 0; j < CV_HAAR_FEATURE_MAX; j++ )
        {
            if( rect[j].weight == 0.0F )
                break;
            CV_TILTED_OFFSETS( fastRect[j].p0, fastRect[j].p1, fastRect[j].p2, fastRect[j].p3, rect[j].r, offset )
        }
    }
}

void CvHaarEvaluator::Feature::write( FileStorage &fs ) const
{
    fs << CC_RECTS << "[";
    for( int ri = 0; ri < CV_HAAR_FEATURE_MAX && rect[ri].r.width != 0; ++ri )
    {
        fs << "[:" << rect[ri].r.x << rect[ri].r.y <<
            rect[ri].r.width << rect[ri].r.height << rect[ri].weight << "]";
    }
    fs << "]" << CC_TILTED << tilted;
}
