/*
 * ShootingStationarySchrodingerSolver.cpp
 *
 *  Created on: 16.02.2013
 *      Author: 
 */

#include "ShootingStationarySchrodingerSolver.h"
#include "../utilSolvers/NumerovCauchySolver.h"
#include "math.h"
#include "OmegaFunction.h"
#include <vector>
#include <cstdio>
#include <sstream>
#include <fstream>

data_t max(data_t x, data_t y) {
	return x > y ? x : y;
}

ShootingStationarySchrodingerSolver::ShootingStationarySchrodingerSolver(RootSolver& _rootSolver, data_t _cauchyPrecision, std::ostream& _logStream) : rootSolver(_rootSolver), cauchyPrecision(_cauchyPrecision), logStream(_logStream){}

void ShootingStationarySchrodingerSolver::setCauchyPrecision(data_t _cauchyPrecision) {
	cauchyPrecision = _cauchyPrecision;
}

void ShootingStationarySchrodingerSolver::logPhi(std::string& phiPlusLogName, std::string& phiMinusLogName, std::vector<data_t>& phiPlusX, std::vector<data_t>& phiMinusX, std::vector<data_t>& phiPlusY, std::vector<data_t>& phiMinusY, data_t energy) {
	std::stringstream stream;
	stream << phiPlusLogName;
	stream << energy;
	std::cout << stream.str();
	std::fstream phiPlusLog(stream.str().c_str(), std::fstream::out);
	stream.str("");
	stream << phiMinusLogName;
	stream << energy;
	std::cout << stream.str();
	std::fstream phiMinusLog(stream.str().c_str(), std::fstream::out);
	for (size_t i = 0; i < phiMinusX.size(); ++i) {
		phiMinusLog << phiMinusX.at(i) << " " << phiMinusY.at(i) << std::endl;
	}
	for (size_t i = 0; i < phiPlusX.size(); ++i) {
		phiPlusLog << phiPlusX.at(i) << " " << phiPlusY.at(i) << std::endl;
	}
	phiPlusLog.close();
	phiMinusLog.close();
}

void ShootingStationarySchrodingerSolver::solve(PotentialFunction & potential, SchrodingerShootingBoundaryConditionGenerator & startCond, 
	SchrodingerShootingBoundaryConditionGenerator & endCond, data_t energyMin, data_t energyMax, data_t energyPrecision, data_t omegaPrecision, data_t energyStep, data_t* eigenValues, 
	int eigenValuesNum, data_t mass, data_t startValue, data_t endValue, data_t xMin, data_t xStep, data_t expandingMultiplier, std::string& phiPlusLogName, std::string& phiMinusLogName) {
	data_t* nextEigenValues = new data_t[eigenValuesNum];
	data_t* currentEigenValues = new data_t[eigenValuesNum];
	for (int i = 0; i < eigenValuesNum; ++i) {
		currentEigenValues[i] = 0;
	}
	data_t maxEigenDiff = 0;
	do {
		maxEigenDiff = 0;
		data_t firstStep = energyPrecision;
		data_t step = energyStep;
		data_t eFirst = energyMin;
		data_t eSecond = energyMin + firstStep;
		std::vector<data_t> omegaValues;
		data_t xSize = xMin + xStep * energyMin;
		int pointsCount = xSize * 2 / cauchyPrecision;
		data_t energyThreshold = max(potential.at(-xSize), potential.at(xSize));
		int currentEigenNum = 0;
		int currentPointNum = 0;
		if (!startCond.isValid(potential, eFirst, -xSize) || !endCond.isValid(potential, eFirst, xSize)) {
			printf("Failure to initialte shooting solver: first energy value is over the potential.\n");
			for (int i = 0; i < eigenValuesNum; ++i) {
				eigenValues[i] = 0;
			}
			return;
		}
		std::vector<data_t> phiPlusX;
		std::vector<data_t> phiPlusY;
		std::vector<data_t> phiMinusX;
		std::vector<data_t> phiMinusY;
		OmegaFunction omegaFun(potential, xMin, xStep, pointsCount, mass, startValue, endValue, startCond, endCond, phiPlusX, phiMinusX, phiPlusY, phiMinusY);
		omegaValues.push_back(omegaFun.at(eFirst));
		int iterationNum = 0;
		//calculate eigen values
		printf("iteration num = %d\n", iterationNum);
		printf("first omega value: %10.20f\n", omegaValues[currentPointNum]);
		if (abs(omegaValues[currentPointNum]) < omegaPrecision) {
			logPhi(phiPlusLogName, phiMinusLogName, phiPlusX, phiMinusX, phiPlusY, phiMinusY, eFirst);
			printf("Root guessed\n");
			nextEigenValues[currentEigenNum] = eFirst;
			++currentEigenNum;
		}
		++currentPointNum;
		while (currentEigenNum < eigenValuesNum && startCond.isValid(potential, eFirst, -xSize) && endCond.isValid(potential, eFirst, xSize)) {
			++iterationNum;
			omegaValues.push_back(omegaFun.at(eSecond));
			printf("iteration num = %d\n", iterationNum);
			printf("omega values: %10.20f, %10.20f\n", omegaValues[currentPointNum - 1], omegaValues[currentPointNum]);
			if (abs(omegaValues[currentPointNum]) < omegaPrecision) {
				logPhi(phiPlusLogName, phiMinusLogName, phiPlusX, phiMinusX, phiPlusY, phiMinusY, eSecond);
				printf("Root guessed\n");
				//getchar();
				nextEigenValues[currentEigenNum] = eSecond;
				++currentEigenNum;
			} else if (omegaValues[currentPointNum] * omegaValues[currentPointNum - 1] <= 0) {
				//root localized
				printf("root localized\n");
				data_t energyValue = rootSolver.solveLocalized(omegaFun, energyPrecision, eFirst, eSecond, omegaValues[currentPointNum - 1], omegaValues[currentPointNum]);
				std::cout << energyValue << std::endl;
				//getchar();
				logPhi(phiPlusLogName, phiMinusLogName, phiPlusX, phiMinusX, phiPlusY, phiMinusY, energyValue);
				nextEigenValues[currentEigenNum] = energyValue;
				++currentEigenNum;
			}
			eFirst = eSecond;
			eSecond += step;
			++currentPointNum;
		}
		//fill out missing energy levels with some marker data
		if (currentEigenNum < eigenValuesNum) {
			for (int i = currentEigenNum; i < eigenValuesNum; ++i) {
				printf("eigen value %d is copied from the previous launch\n", i);
				nextEigenValues[i] = currentEigenValues[i];
			}
		}
		//calculate eigen diff
		printf("Eigen values:\n");
		for (int i = 0; i < eigenValuesNum; ++i) {
			printf("%10.10f \n", nextEigenValues[i]);
			maxEigenDiff = max(maxEigenDiff, abs(currentEigenValues[i] - nextEigenValues[i]));
			currentEigenValues[i] = nextEigenValues[i];
		}
		if (expandingMultiplier == 0) {
			break;
		}
		//energyPrecision /= 10;
		xStep *= expandingMultiplier;
		xMin *= expandingMultiplier;
	} while (maxEigenDiff > energyPrecision);

	for (int i = 0; i < eigenValuesNum; ++i) {
		eigenValues[i] = currentEigenValues[i];
	}

	delete[] currentEigenValues;
	delete[] nextEigenValues;
}