/*
 * Matrix.h
 *
 *  Created on: 30.01.2013
 *      Author: 
 */

#ifndef MATRIX_H_
#define MATRIX_H_

#include "../common/types.h"
#include <iostream>
#include <vector>
#include <cstdio>
#include <cstdlib>

template<typename data_t>
class Matrix {
public:
	Matrix(int _n, int _m);
	Matrix(Matrix<data_t> const & arg);
	~Matrix();
	Matrix<data_t> inverse();
	data_t getData(int row, int column) const;
	void setData(int row, int column, data_t value);
	Matrix<data_t> operator*(Matrix<data_t> const & rArg) const;
	Matrix<data_t> operator+(Matrix<data_t> const& rArg) const;
	Matrix<data_t> operator-(Matrix<data_t> const& rArg) const;
	Matrix<data_t>& operator=(Matrix<data_t> const& rhs);
	int getWidth() const;
	int getHeight() const;
	Matrix<data_t> getSubMatrix(std::vector<int>& rows,
			std::vector<int>& columns);
	Matrix<data_t> getSubMatrix(int fromRow, int toRow, int fromColumn, int toColumn);
	void setSubMatrix(std::vector<int>& rows, std::vector<int>& columns,
			Matrix<data_t>& subMatrix);
	void setSubMatrix(int fromRow, int toRow, int fromColumn, int toColumn, Matrix<data_t>& subMatrix);
	void swapRows(int row1, int row2);
	int getNonZeroColumn(int startColumn, int startRow, int& nonZeroRow);
	Matrix<data_t> concat(Matrix<data_t>& m2);
	void multiplyRow(int row, data_t mul);
	void addRow(int to, int from, data_t mul);
	void jordan();
	static Matrix<data_t> getIdentity(int order);
	static Matrix<data_t> getZero(int _maxRows, int _maxColumns);
	static Matrix<data_t> getDiagonalMatrix(std::vector<data_t> diagonal);
private:
	void clearColumn(int startRow, int column);
	Matrix jordanInverse();
	int maxRows;
	int maxColumns;
	data_t** data;
};

template<typename data_t>
std::ostream& operator<<(std::ostream &os, Matrix<data_t> const &r);

template<typename data_t>
Matrix<data_t>::Matrix(int _n, int _m) :
		maxRows(_n), maxColumns(_m) {
	data = new data_t*[maxRows];
	for (int i = 0; i < maxRows; ++i) {
		data[i] = new data_t[maxColumns];
	}
}
template<typename data_t>
Matrix<data_t>::Matrix(Matrix<data_t> const & arg) :
		maxRows(arg.maxRows), maxColumns(arg.maxColumns) {
	data = new data_t*[maxRows];
	for (int i = 0; i < maxRows; ++i) {
		data[i] = new data_t[maxColumns];
		for (int j = 0; j < maxColumns; ++j) {
			data[i][j] = arg.data[i][j];
		}
	}
}
template<typename data_t>
void Matrix<data_t>::setSubMatrix(std::vector<int>& rows,
		std::vector<int>& columns, Matrix& subMatrix) {
	int row = 0;
	int column = 0;
	for (std::vector<int>::iterator rowIt = rows.begin(); rowIt != rows.end();
			++rowIt, ++row) {
		column = 0;
		for (std::vector<int>::iterator columnIt = columns.begin();
				columnIt != columns.end(); ++columnIt, ++column) {
			setData(*rowIt, *columnIt, subMatrix.getData(row, column));
		}
	}
}
template<typename data_t>
void Matrix<data_t>::setSubMatrix(int fromRow, int toRow, int fromColumn, int toColumn, Matrix<data_t>& subMatrix) {
	std::vector<int> rowsV;
	std::vector<int> columnsV;
	int  rows = toRow - fromRow;
	int columns = toColumn - fromColumn;
	for (int i = 0; i < rows; ++i) {
		rowsV.push_back(i + fromRow);
	}
	for (int i = 0; i < columns; ++i) {
		columnsV.push_back(i + fromColumn);
	}
	setSubMatrix(rowsV, columnsV, subMatrix);
}
/**
 * Creates submatrix with data from given rows and columns in given order.
 */
template<typename data_t>
Matrix<data_t> Matrix<data_t>::getSubMatrix(std::vector<int>& rows,
		std::vector<int>& columns) {
	Matrix result(rows.size(), columns.size());
	int row = 0;
	int column = 0;
	for (std::vector<int>::iterator rowIt = rows.begin(); rowIt != rows.end();
			++rowIt, ++row) {
		column = 0;
		for (std::vector<int>::iterator columnIt = columns.begin();
				columnIt != columns.end(); ++columnIt, ++column) {
			result.setData(row, column, getData(*rowIt, *columnIt));
		}
	}
	return result;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::getSubMatrix(int fromRow, int toRow, int fromColumn, int toColumn) {
	int height = toRow-fromRow;
	int width = toColumn-fromColumn;
	Matrix result(height, width);
	int row = 0;
	int column = 0;
	for (int row = 0; row < height; ++row) {
		for (int column = 0; column < width; ++column) {
			result.setData(row, column, getData(fromRow + row, fromColumn + column));
		}
	}
	return result;
}
template<typename data_t>
void Matrix<data_t>::swapRows(int row1, int row2) {
	if (row1 != row2) {
		data_t * buf = data[row1];
		data[row1] = data[row2];
		data[row2] = buf;
	}
}
template<typename data_t>
int Matrix<data_t>::getNonZeroColumn(int startColumn, int startRow,
		int& nonZeroRow) {
	for (int column = startColumn; column < maxColumns; ++column) {
		for (int row = startRow; row < maxRows; ++row) {
			if (getData(row, column) != 0.0) { //XXX: replace with abs check
				nonZeroRow = row;
				return column;
			}
		}
	}
	nonZeroRow = -1;
	return -1;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::concat(Matrix& m2) {
	if (maxRows != m2.maxRows) {
		//XXX: error
		fprintf(stderr, "concat error: %d rows expected, but %d rows are found",maxRows, m2.maxRows);
		exit(331);
	}
	Matrix result(maxRows, maxColumns + m2.maxColumns);
	for (int row = 0; row < maxRows; ++row) {
		for (int column = 0; column< maxColumns; ++column) {
			result.setData(row, column, getData(row, column));
		}
		for (int column = 0; column < m2.maxColumns; ++column) {
			result.setData(row, column + maxColumns, m2.getData(row, column));
		}
	}
	return result;
}
template<typename data_t>
void Matrix<data_t>::multiplyRow(int row, data_t mul) {
	for (int column = 0; column < maxColumns; ++column) {
		setData(row, column, getData(row, column) * mul);
	}
}
template<typename data_t>
void Matrix<data_t>::addRow(int to, int from, data_t mul) {
	for (int column = 0; column < maxColumns; ++column) {
		setData(to, column, getData(to, column) + getData(from, column) * mul);
	}
}
template<typename data_t>
void Matrix<data_t>::clearColumn(int startRow, int column) {
	for (int row = 0; row < maxRows; ++row) {
		if (row == startRow) {
			continue;
		}
		addRow(row, startRow, -getData(row, column));
	}
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::getIdentity(int order) {
	Matrix result(order, order);
	for (int row = 0; row < order; ++row) {
		for (int column = 0; column < order; ++column) {
			result.setData(row, column, row == column ? 1 : 0);
		}
	}
	return result;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::getZero(int _maxRows, int _maxColumns) {
	Matrix result(_maxRows, _maxColumns);
	for (int row = 0; row < _maxRows; ++row) {
		for (int column = 0; column < _maxColumns; ++column) {
			result.setData(row, column, 0);
		}
	}
	return result;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::getDiagonalMatrix(std::vector<data_t> diagonal) {
	size_t order = diagonal.size();
	Matrix result(order, order);
	for (size_t row = 0; row < order; ++row) {
		for (size_t column = 0; column < order; ++column) {
			result.setData(row, column, row == column ? diagonal.at(row) : 0);
		}
	}
	return result;
}
template<typename data_t>
void Matrix<data_t>::jordan() {
	int currentColumn = 0;
	int currentRow = 0;
	int nonZeroRow = 0;
	while (currentColumn < maxColumns && currentRow < maxRows) {
		currentColumn = getNonZeroColumn(currentColumn, currentRow, nonZeroRow);
		swapRows(currentRow, nonZeroRow);
		multiplyRow(currentRow, 1.0 / getData(currentRow, currentColumn));
		clearColumn(currentRow, currentColumn);
		++currentRow;
		++currentColumn;
	}
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::jordanInverse() {
	Matrix<data_t> id = getIdentity(maxRows);
	Matrix<data_t> result = concat(id);
	result.jordan();
	std::vector<int> columns;
	std::vector<int> rows;
	for (int row = 0; row < maxRows; ++row) {
		rows.push_back(row);
	}
	for (int column = 0; column < id.maxColumns; ++column) {
		columns.push_back(column + maxColumns);
	}
	return result.getSubMatrix(rows, columns);
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::inverse() {
	if (maxRows == maxColumns) {
		return jordanInverse();
	} else {
		return Matrix<data_t>(maxRows, maxColumns); //XXX: insert something here
	}
}
template<typename data_t>
Matrix<data_t>::~Matrix() {
	for (int i = 0; i < maxRows; ++i) {
		delete[] (data[i]);
	}
	delete[] data;
}
template<typename data_t>
data_t Matrix<data_t>::getData(int row, int column) const {
	return data[row][column];
}
template<typename data_t>
void Matrix<data_t>::setData(int row, int column, data_t value) {
	data[row][column] = value;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::operator*(Matrix<data_t> const & rArg) const {
	Matrix<data_t> res(maxRows, rArg.maxColumns);
	for (int outRow = 0; outRow < maxRows; ++outRow) {
		for (int outColumn = 0; outColumn < rArg.maxColumns; ++outColumn) {
			data_t sum = 0;
			for (int i = 0; i < maxColumns; ++i) {
				sum += data[outRow][i] * rArg.getData(i, outColumn);
			}
			res.setData(outRow, outColumn, sum);
		}
	}
	return res;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::operator+(Matrix<data_t> const& rArg) const {
	Matrix<data_t> res(*this);
	for (int row = 0; row < maxRows; ++row) {
		for (int column = 0; column < maxColumns; ++column) {
			res.setData(row, column, getData(row, column) + rArg.getData(row, column));
		}
	}
	return res;
}
template<typename data_t>
Matrix<data_t> Matrix<data_t>::operator-(Matrix<data_t> const& rArg) const {
	Matrix<data_t> res(*this);
	for (int row = 0; row < maxRows; ++row) {
		for (int column = 0; column < maxColumns; ++column) {
			res.setData(row, column, getData(row, column) - rArg.getData(row, column));
		}
	}
	return res;
}
template<typename data_t>
Matrix<data_t>& Matrix<data_t>::operator=(Matrix const& rhs) {
	for (int i = 0; i < maxRows; ++i) {
		delete[] (data[i]);
	}
	delete[] data;
	maxRows = rhs.maxRows;
	maxColumns = rhs.maxColumns;
	data = new data_t*[maxRows];
	for (int i = 0; i < maxRows; ++i) {
		data[i] = new data_t[maxColumns];
		for (int j = 0; j < maxColumns; ++j) {
			data[i][j] = rhs.data[i][j];
		}
	}
	return *this;
}
template<typename data_t>
int Matrix<data_t>::getHeight() const {
	return maxRows;
}
template<typename data_t>
int Matrix<data_t>::getWidth() const {
	return maxColumns;
}
template<typename data_t>
std::ostream& operator<<(std::ostream &os, Matrix<data_t> const &r) {
	for (int i = 0; i < r.getHeight(); ++i) {
		for (int j = 0; j < r.getWidth(); ++j) {
			os << r.getData(i, j) << " ";
		}
		os << std::endl;
	}
	os << std::endl;
	return os;
}

#endif /* MATRIX_H_ */
