/**
 * @author Malkovsky Nikolay
 * @brief Demonstration program for automated timetable constructing.
 *
 * @date 15.05.2013
 */
#include <algorithm>
#include <ctime>
#include <cstdlib>
#include <cmath>
#include <iostream>
#include <cstdio>
#include <map>
#include <cassert>
#include <cstring>
#include <vector>
#include <string>

using namespace std;

const int inf = 1 << 30;
const int DAYS = 6;
const int LESSONS_PER_DAY = 4;
/**
 * This const determines number of lectors/groups with single lesson to schedule.
 * That is the theoretically minimun number of penalty function
 */
const int single_day = 3;
/**
 * Each color are actually correspond to certain timeslot, term "color" is
 * used as from original algorithm.
 * COLORS constant determines overall number of timeslots abailable for assignment.
 */        
const int COLORS = LESSONS_PER_DAY * DAYS;

/*==========================================================================================*/

/**
 * Structure representing audience type. 
 * Considering every lesson requires specific auditory type.
 */
struct Audience_type {
	int id;
	string name;
};

/**
 * Structure representing audience.
 */
struct Audience {
	int type;
	string name;
};

/**
 * Structure representing lesson.
 * Lesson is triple Lector, group(s) and Subject name
 * TODO: Lessons to fill more then one timeslot and lessons for more then one group
 */
struct Lesson {
	int lector_id;
	int group_id;
	string subject;
	Lesson(int l, int g, string s) {
		lector_id = l;
		group_id = g;
		subject = s;
	}
};


/**
 * Structure representing schedule.
 */
struct Schedule {
	int** lectors_schedule;
	int** groups_schedule;
	bool* fixed_time;
	int* assigned_time;
	int* deg_lectors;
	int* deg_groups;
};

/**
 * Number of lectors.
 * TODO: rename
 */
int n;
/**
 * Arrays for lesson graph representation
 */
int* head;
int* next;
bool* visited;

/**
 * Inner representation data.
 * TODO: create a class for this data.
 */
map<string, int> groups;
vector<string> name;
vector<Lesson*> lessons;
vector<string> groups_num;
map<int, int> audience_number;


/*==========================================================================================*/

/**
 * Initializing schedule - allocating memory for internal data structures
 * and initializing their values.
 */
Schedule* initialize_schedule() {
	Schedule* _schedule = new Schedule();
	_schedule->lectors_schedule = new int*[n];
    _schedule->groups_schedule = new int*[groups_num.size()];
    _schedule->deg_lectors = new int[n];
    _schedule->deg_groups = new int[groups_num.size()];
    for(int i = 0; i < n; ++i) {
    	_schedule->deg_lectors[i] = 0;
    }
    for(int i = 0; i < groups_num.size(); ++i) {
    	_schedule->deg_groups[i] = 0;
    }
    
    for(int i = 0; i < n; ++i) {
    	_schedule->lectors_schedule[i] = new int[COLORS];
    	for(int j = 0; j < COLORS; ++j) {
    		_schedule->lectors_schedule[i][j] = -1;
    	}
    }

    for(int i = 0; i < groups_num.size(); ++i) {
    	_schedule->groups_schedule[i] = new int[COLORS];
    	for(int j = 0; j < COLORS; ++j) {
    		_schedule->groups_schedule[i][j] = -1;
    	}
    }
    _schedule->fixed_time = new bool[lessons.size()];
    _schedule->assigned_time = new int[lessons.size()];
    for(int i = 0; i < lessons.size(); ++i) {
		_schedule->fixed_time[i] = false;
		_schedule->assigned_time[i] = -1;
		_schedule->deg_lectors[lessons[i]->lector_id]++;
    	_schedule->deg_groups[lessons[i]->group_id]++;
    }
    return _schedule;
}

/**
 * Standart dfs with 3 modifications.
 * On stage 0 only edges that connects two verticies with max degree are considered
 * On stage 1 only edges that incindent vertex with max degree are considered
 * On stage 2 all the edges are considered.
 * UPD: different stages works slightly longer when combined with annealing.
 */
bool try_add(Schedule* schedule, int v, int color, int stage, int max_deg) {
	if(visited[v]) {
		return false;
	}
	visited[v] = true;
	for(int i = head[v]; i != -1; i = next[i]) {
		//if(((max_deg > schedule->deg_lectors[v]) + (max_deg > schedule->deg_groups[lessons[i]->group_id])) > stage) {
		//	continue;
		//}
		if(schedule->assigned_time[i] != -1) {
			continue;
		}
		if(schedule->groups_schedule[lessons[i]->group_id][color] == -1) {
			schedule->deg_groups[lessons[i]->group_id]--;
            schedule->groups_schedule[lessons[i]->group_id][color] = i;
            schedule->lectors_schedule[lessons[i]->lector_id][color] = i;
            schedule->assigned_time[i] = color;
            return true;
		}
		if(!schedule->fixed_time[schedule->groups_schedule[lessons[i]->group_id][color]] &&
				try_add(schedule, lessons[schedule->groups_schedule[lessons[i]->group_id][color]]->lector_id, color, stage, max_deg)) {
			schedule->assigned_time[schedule->groups_schedule[lessons[i]->group_id][color]] = -1;
			schedule->groups_schedule[lessons[i]->group_id][color] = i;
        	schedule->lectors_schedule[lessons[i]->lector_id][color] = i;
        	schedule->assigned_time[i] = color;
            return true;
        } 
	}
	return false;
}

/**
 * Function counting overall number of "windows" for lectors and groups for specific schedule.
 * Window is defined as free time slot for lector/group when there is occupied slots of the
 * same day after and before the window.
 */
int window_count(Schedule* schedule) {
	int result = 0;
	for(int j = 0; j < n; ++j) {
    	int windows = 0;
    	for(int i = 0; i < DAYS; ++i) {
    		int mn = -1, mx = -1, num = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->lectors_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				if(mn == -1) mn = k;
    				mx = k;
    				num++;
    			}
    		}
    		if(mn != -1) windows += mx - mn + 1 - num;
    	}
    	result += windows;
    }

    for(int j = 0; j < groups_num.size(); ++j) {
    	int windows = 0;
    	for(int i = 0; i < DAYS; ++i) {
    		int mn = -1, mx = -1, num = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->groups_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				if(mn == -1) mn = k;
    				mx = k;
    				num++;
    			}
    		}
    		if(mn != -1) windows += mx - mn + 1 - num;
    	}
    	result += windows;
    }
    return result;
}

/**
 * Number of days with single lesson for each lector/group.
 */
int days_with_single_lesson(Schedule* schedule) {
	int result = 0;
	for(int j = 0; j < n; ++j) {
    	for(int i = 0; i < DAYS; ++i) {
    		int cnt = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->lectors_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				cnt++;
    			}
    		}
    		if(cnt == 1) result++;
    	}
    }

    for(int j = 0; j < groups_num.size(); ++j) {
    	int windows = 0;
    	for(int i = 0; i < DAYS; ++i) {
    		int cnt = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->groups_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				cnt++;
    			}
    		}
    		if(cnt == 1) result++;
    	}
    }
    return result;
}
/**
 * Penalty function used in annealing.
 * For this particular instance penalty function is 10 * (number of unassigned lessons) + 
 *	overall window number + days with single lesson (for each lector/group)
 */

int penalty(Schedule* schedule) {
	int assigned_count = 0;
	for(int i = 0; i < lessons.size(); ++i) {
		assigned_count += schedule->assigned_time[i] == -1;
	}
	return 10 * assigned_count + window_count(schedule) + days_with_single_lesson(schedule) - single_day;
}

/**
 * Copies content of one schedule to another.
 */
void copy_schedule(Schedule* from, Schedule* to) {
	for(int i = 0; i < n; ++i) {
		to->deg_lectors[i] = from->deg_lectors[i];
		for(int c = 0; c < COLORS; ++c) {
			to->lectors_schedule[i][c] = from->lectors_schedule[i][c];
        }
	}
	for(int i = 0; i < groups_num.size(); ++i) {
		to->deg_lectors[i] = from->deg_lectors[i];
		for(int c = 0; c < COLORS; ++c) {
			to->groups_schedule[i][c] = from->groups_schedule[i][c];
        }
	}
	for(int i = 0; i < lessons.size(); ++i) {
		to->assigned_time[i] = from->assigned_time[i];
		to->fixed_time[i] = from->fixed_time[i];
	}
}

/**
 * Applies general graph coloring algorithms to constrct a schedule.
 * UPD: Stages are removed to increase performance slightly
 * UPD: When combined with annealing, vertecies with all edges colored
 * are slowing algorithm perfomance significantly, so they are removed from
 * consideration.
 */
int build_schedule(Schedule* schedule) {
	int res = 0;
	for(int color = 0; color < COLORS; ++color) {
		int max_deg = 0;
		for(int i = 0; i < n; ++i) {
			max_deg = max(max_deg, schedule->deg_lectors[i]);
		}
		for(int i = 0; i < groups_num.size(); ++i) {
			max_deg = max(max_deg, schedule->deg_groups[i]);
		}
		if(max_deg == 0) {
			break;
		}
		//for(int stage = 0; stage < 3; ++stage) {
        	for(int i = 0; i < n; ++i) {
    			if(schedule->lectors_schedule[i][color] != -1 || !schedule->deg_lectors[i]) continue;
    			memset(visited, 0, sizeof(bool) * lessons.size());
    				
    			if(try_add(schedule, i, color, 2, max_deg)) {
    				schedule->deg_lectors[i]--;
    			}
    		}
    	//}
    }
    for(int i = 0; i < lessons.size(); ++i) {
    	if(schedule->assigned_time[i] == -1) res++;
    }
    return res;
}

/**
 * Unassignes lesson l in schedule.
 */
inline void unassign(Schedule* schedule, int l) {
	if(schedule->assigned_time[l] == -1) {
		return;
	}
	schedule->deg_lectors[lessons[l]->lector_id]++;
	schedule->deg_groups[lessons[l]->group_id]++;
	schedule->groups_schedule[lessons[l]->group_id][schedule->assigned_time[l]] = -1;
	schedule->lectors_schedule[lessons[l]->lector_id][schedule->assigned_time[l]] = -1;
	schedule->assigned_time[l] = -1;
}

/**
 * Assignes time t for lesson l in schedule. 
 */
inline void assign(Schedule* schedule, int l, int t) {
	if(schedule->lectors_schedule[lessons[l]->lector_id][t] != -1) {
		unassign(schedule, schedule->lectors_schedule[lessons[l]->lector_id][t]);
	}
	if(schedule->groups_schedule[lessons[l]->group_id][t] != -1) {
		unassign(schedule, schedule->groups_schedule[lessons[l]->group_id][t]);
	}
	unassign(schedule, l);
    schedule->deg_lectors[lessons[l]->lector_id]--;
	schedule->deg_groups[lessons[l]->group_id]--;
	schedule->groups_schedule[lessons[l]->group_id][t] = l;
	schedule->lectors_schedule[lessons[l]->lector_id][t] = l;
	schedule->assigned_time[l] = t;
}

/**
 * Assignes time and forbids to change it unless unfix_time is used.
 */
void fix_time(Schedule* schedule, int lesson_num, int t) {
	assign(schedule, lesson_num, t);
	schedule->fixed_time[lesson_num] = true;
}

/**
 * Opposite of fix_time.
 */
void unfix_time(Schedule* schedule, int lesson_num, int t) {
	schedule->fixed_time[lesson_num] = false;
}


/**
 * Constructs schedule using simulated annealing method.
 * Param steps is used to bound number of iteration of annealing.
 * 
 * Annealing parameters are NOT optimal. Still work fine for considered test cases.
 * Does NOT use graph coloring between iterations.
 */
int annealing(Schedule* schedule, int steps) {
	Schedule* temp_schedule = initialize_schedule();
	//build_schedule(schedule);
	srand(time(NULL));
	double K = lessons.size();
	double factor = 1.1;
	int t = 1;

	while(steps --> 0) {
		int pen = penalty(schedule);
		//cerr << pen << endl;
		copy_schedule(schedule, temp_schedule);
		int l = rand() % lessons.size();
        int cur = 0;
        while(l) {
        	if(!schedule->fixed_time[cur]) l--;
        	cur = (cur + 1) % lessons.size();
        }
        while(schedule->fixed_time[cur]) cur = (cur + 1) % lessons.size();
        int c = rand() % COLORS;
        int col = 0;
        while(c) {
        	if(!(schedule->lectors_schedule[lessons[cur]->lector_id][col] != -1 && schedule->fixed_time[schedule->lectors_schedule[lessons[cur]->lector_id][col]]) && 
        		!(schedule->groups_schedule[lessons[cur]->group_id][col] != -1 && schedule->fixed_time[schedule->groups_schedule[lessons[cur]->group_id][col]])) c--;
        	col = (col + 1) % COLORS;
        }
        while((schedule->lectors_schedule[lessons[cur]->lector_id][col] != -1 && schedule->fixed_time[schedule->lectors_schedule[lessons[cur]->lector_id][col]]) || 
        	(schedule->groups_schedule[lessons[cur]->group_id][col] != -1 && schedule->fixed_time[schedule->groups_schedule[lessons[cur]->group_id][col]])) col = (col + 1) % COLORS;
        assign(schedule, cur, col);
        
		//build_schedule(schedule);
		int new_pen = penalty(schedule);

		if(new_pen == 0) {
			cerr << "Done in " << t << " steps" << endl;
			break;
		}
		if(new_pen < pen) {
			continue;
		}
		if(rand() > exp((pen - new_pen) / K) * RAND_MAX) {
			copy_schedule(temp_schedule, schedule);
		}
		t++;
		K /= factor;
	}
	int res = 0;
    for(int i = 0; i < lessons.size(); ++i) {
    	if(schedule->assigned_time[i] == -1) res++;
    }
    return res; 
}

/**
 * Reads info from standart input.
 * Format is:
 * Number of lectors on the first line folowed by series of lectors descriptions.
 * Each description consistes lector name and number of different lessons on a single line followed by
 * lessons descriptions. Each lesson description consist of number of times per week, group name and subject name.
 */
void read_info() {
	scanf("%d", &n);
    //
    for(int i = 0; i < n; ++i) {
        int k;
    	string str;
    	cin >> str >> k;
    	name.push_back(str);
    	for(int j = 0; j < k; ++j) {
    		int hours;
    		string group_idx, subject_name;
    		cin >> hours >> group_idx >> subject_name;
    		if(groups.find(group_idx) == groups.end()) {
    			groups.insert(make_pair(group_idx, groups_num.size()));
    			groups_num.push_back(group_idx); 
    		}
    		for(int l = 0; l < hours; ++l) {
    			lessons.push_back(new Lesson(i, groups[group_idx], subject_name));
    		}
    	}
    }
}               

/**
 * Memory allocation and inialization of inner representation structure.
 */
void initialize_inner_data() {
	visited = new bool[lessons.size()];
    head = new int[n];
    next = new int[lessons.size()];
    for(int i = 0; i < n; ++i) {
    	head[i] = -1;
    }
    for(int i = 0; i < lessons.size(); ++i) {
    	next[i] = head[lessons[i]->lector_id];
    	head[lessons[i]->lector_id] = i;
    }
}   

int main() {
    read_info();
    initialize_inner_data();
    Schedule* schedule = initialize_schedule();
    fix_time(schedule, 0, 3);
    fix_time(schedule, 1, 4);
    cerr << "Lessons to schedule: " << lessons.size() << endl;
    //cerr << "Failed to schedule " << build_schedule(schedule) << " lessons" << endl;
   	int start = clock();
    int failed = annealing(schedule, 5000000);
    cerr << (double)(clock() - start) / CLOCKS_PER_SEC << endl;;
    cerr << "Failed to schedule " << failed << " lessons" << endl;

    /*
     * Outputs schedule table grouped by lectors
     */
    for(int j = 0; j < n; ++j) {
    	for(int i = 0; i < COLORS; ++i) {
    		if(schedule->lectors_schedule[j][i] != -1) {
    			cout << name[lessons[schedule->lectors_schedule[j][i]]->lector_id] << " " << groups_num[lessons[schedule->lectors_schedule[j][i]]->group_id]
    				<< " " << lessons[schedule->lectors_schedule[j][i]]->subject << " ";
    			switch(i / 4) {
    				case 0: cout << "Monday "; break;
    				case 1: cout << "Tuesday "; break;
    				case 2: cout << "Wednesday "; break;
    				case 3: cout << "Thursday "; break;
    				case 4: cout << "Friday "; break;
    				case 5: cout << "Saterday "; break;
    			}
    			cout << 1 + (i % 4) << endl;
    		}
    	}
    	cout << endl;
    }

    cout << "=======================================================" << endl;
    /*
     * Outputs schedule table grouped by groups
     */
    for(int j = 0; j < groups_num.size(); ++j) {
    	for(int i = 0; i < COLORS; ++i) {
    		if(schedule->groups_schedule[j][i] != -1) {
    			cout << name[lessons[schedule->groups_schedule[j][i]]->lector_id] << " " << groups_num[lessons[schedule->groups_schedule[j][i]]->group_id]
    				<< " " << lessons[schedule->groups_schedule[j][i]]->subject << " ";
    			switch(i / 4) {
    				case 0: cout << "Monday "; break;
    				case 1: cout << "Tuesday "; break;
    				case 2: cout << "Wednesday "; break;
    				case 3: cout << "Thursday "; break;
    				case 4: cout << "Friday "; break;
    				case 5: cout << "Saterday "; break;
    			}
    			cout << 1 + (i % 4) << endl;
    		}
    	}
    	cout << endl;
    }

    /*
     * printing "windows" count for all the lectors/groups
     */
    for(int j = 0; j < n; ++j) {
    	int windows = 0;
    	for(int i = 0; i < DAYS; ++i) {
    		int mn = -1, mx = -1, num = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->lectors_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				if(mn == -1) mn = k;
    				mx = k;
    				num++;
    			}
    		}
    		if(mn != -1) windows += mx - mn + 1 - num;
    	}
    	cout << name[j] << " " << windows;
    	cout << endl;
    }

    for(int j = 0; j < groups_num.size(); ++j) {
    	int windows = 0;
    	for(int i = 0; i < DAYS; ++i) {
    		int mn = -1, mx = -1, num = 0;
    		for(int k = 0; k < LESSONS_PER_DAY; ++k) {
    			if(schedule->groups_schedule[j][LESSONS_PER_DAY * i + k] != -1) {
    				if(mn == -1) mn = k;
    				mx = k;
    				num++;
    			}
    		}
    		if(mn != -1) windows += mx - mn + 1 - num;
    	}
    	cout << groups_num[j] << " " << windows;
    	cout << endl;
    }
           

	return 0;
} 

