package communities.parseciteceer;

import communitiesGraph.Author;
import communitiesGraph.Paper;
import org.neo4j.graphdb.*;
import org.neo4j.graphdb.factory.GraphDatabaseFactory;
import org.neo4j.graphdb.index.Index;
import org.neo4j.graphdb.traversal.Evaluators;
import org.neo4j.graphdb.traversal.TraversalDescription;
import org.neo4j.kernel.Traversal;

import java.sql.SQLException;
import java.util.*;

/**
 * Created with IntelliJ IDEA.
 * User: nikita
 * Date: 01.03.13
 * Time: 23:20
 * To change this template use File | Settings | File Templates.
 */
public class GraphDB {
    private GraphDatabaseService graphDatabase;
    private Index<Node> nodeIndex;

    private static final String INDEX_NAME = "nodes";
    private static final String DB_PATH = "/usr/local/neo4j-community-1.8.2/citeseer";

    // Index Keys (IK)
    public static final String IK_AUTHOR_ID = "author_id";
    public static final String IK_AUTHOR_NAME = "author_name";
    public static final String IK_PAPER_ID = "paper_id";

    // Property Keys (PK)
    public static final String PK_AUTHOR_ID = "author_id";
    public static final String PK_AUTHOR_NAME = "author";
    public static final String PK_AUTHOR_CITE = "cite";
    public static final String PK_AUTHOR_URL = "url";
    public static final String PK_PAPER_ID = "paper_id";
    public static final String PK_KIND_NODE = "kind";

    public GraphDB() {
        start();
        nodeIndex = graphDatabase.index().forNodes("nodes");
    }

    public void start() {
        graphDatabase = new GraphDatabaseFactory().newEmbeddedDatabase(DB_PATH);
        registerShutdownHook(graphDatabase);
    }

    public void shutdown() {
        graphDatabase.shutdown();
    }

    public void buildFromMySQLDatabase() throws SQLException, Exception {
        Database sqlDB = new Database();
        Transaction tx = graphDatabase.beginTx();
        try {
            // Get all authors from sql base. Create new nodes and indexes by authors.
            ArrayList<Author> authors = sqlDB.getAllAuthors();
            Map<Integer, Node> authorNodeByID = new HashMap<>();
            for (Author author : authors) {
                Node node = createAndIndexAuthor(author);
                if (node != null) {
                    authorNodeByID.put(author.getId(), node);
                }
            }
            System.out.println("Obtained all authors from sql base. And create new nodes and indexes.");

            // Get all papers from sql base. Create new nodes and indexes by papers.
            Collection<Integer> papersID = sqlDB.getAllPapersID();
            Map<Integer, Node> paperNodeByID = new HashMap<>();
            for (int paperID : papersID) {
                Node node = createAndIndexPaper(paperID);
                if (node != null) {
                    paperNodeByID.put(paperID, node);
                }
            }
            System.out.println("Obtained all papers.");

            sqlDB.closeConnection();
            tx.success();
            tx.finish();

            // For each author add relation with his papers, co-authors and citators.
            for (Node authorNode : authorNodeByID.values()) {
                tx = graphDatabase.beginTx();
                if (authorNode.hasRelationship(Direction.OUTGOING)) {
                    tx.finish();
                    continue;
                }
                sqlDB = new Database();
                try {
                    int authorID = (Integer) authorNode.getProperty(PK_AUTHOR_ID);
                    System.out.println("Add relationships for author with id: " + authorID);
                    Collection<Paper> papers = sqlDB.getPapersFromAuthorsID(authorID);
                    for (Paper paper : papers) {
                        Node paperNode = paperNodeByID.get(paper.getId());
                        authorNode.createRelationshipTo(paperNode, RelationType.AUTHORSHIP);

                        addRelationships(authorNode, paper.getAuthors(), RelationType.CO_AUTHORSHIP, authorNodeByID);
                        addRelationships(authorNode, paper.getCitingAuthors(), RelationType.CITATOR, authorNodeByID);
                    }
                    tx.success();
                } finally {
                    sqlDB.closeConnection();
                    tx.finish();
                }
            }

            System.out.println("All relationships are added");
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            sqlDB.closeConnection();
            tx.finish();
        }
    }

    private Node createAndIndexAuthor(Author author) {
        Node node = nodeIndex.get(IK_AUTHOR_ID, author.getId()).getSingle();
        if (node == null) {
            node = graphDatabase.createNode();
            node.setProperty(PK_KIND_NODE, "author");
            node.setProperty(PK_AUTHOR_ID, author.getId());
            node.setProperty(PK_AUTHOR_NAME, author.getName().toLowerCase());
            node.setProperty(PK_AUTHOR_URL, author.getUrl());
            node.setProperty(PK_AUTHOR_CITE, author.getSite());

            nodeIndex.putIfAbsent(node, IK_AUTHOR_ID, author.getId());
            nodeIndex.putIfAbsent(node, IK_AUTHOR_NAME, author.getName());
        }

        return node;
    }

    private Node createAndIndexPaper(int paperID) {
        Node node = nodeIndex.get(IK_PAPER_ID, paperID).getSingle();
        if (node == null) {
            node = graphDatabase.createNode();
            node.setProperty(PK_KIND_NODE, "paper");
            node.setProperty(PK_PAPER_ID, paperID);

            nodeIndex.putIfAbsent(node, IK_PAPER_ID, paperID);
        }
        return node;
    }

    private void addRelationships(
            Node source, Collection<Author> authors, RelationType relationType, Map<Integer, Node> authorNodeByID) {

        //Relationship relationship;
        for (Author otherAuthor : authors) {
            Node otherAuthorNode = authorNodeByID.get(otherAuthor.getId());
            source.createRelationshipTo(otherAuthorNode, relationType);
            if (source.hasRelationship()) {
                System.out.println("  " + source.getProperty(PK_AUTHOR_ID) + " has relationships");
            }
        }

    }

/*    public HashSet<Node> getCoAuthors(Node authorNode) {
        return getRelationAuthors(authorNode, RelationType.CO_AUTHORSHIP, Direction.OUTGOING);
    }

    public HashSet<Node> getCitators(Node authorNode) {
        return getRelationAuthors(authorNode, RelationType.CITATOR, Direction.OUTGOING);
    }*/

    public HashSet<Node> getRelationAuthors(Node authorNode, RelationType relType, Direction direction) {
        Iterable<Relationship> relationships = authorNode.getRelationships(relType, direction);
        HashSet<Node> relationAuthors = new HashSet<>();
        for (Relationship relation : relationships) {
            relationAuthors.add(relation.getEndNode());
        }
        return relationAuthors;
    }

    public Iterable<Node> getCoAuthors(Integer authorID) {
        Node desiredAuthor = nodeIndex.get(IK_AUTHOR_ID, authorID).getSingle();
        return getCoAuthors(desiredAuthor);
    }

    public Iterable<Node> getCoAuthors(Node author) {
        TraversalDescription td = Traversal.description()
                .evaluator(Evaluators.toDepth(1))
                .relationships(RelationType.CO_AUTHORSHIP, Direction.OUTGOING)
                .evaluator(Evaluators.excludeStartPosition());

        return td.traverse(author).nodes();
    }

    public Iterable<Node> getCitators(Integer authorID) {
        Node desiredAuthor = nodeIndex.get(IK_AUTHOR_ID, authorID).getSingle();
        return getCitators(desiredAuthor);
    }

    public Iterable<Node> getCitators(Node author) {
        TraversalDescription td = Traversal.description()
                .evaluator(Evaluators.toDepth(1))
                .relationships(RelationType.CITATOR, Direction.OUTGOING)
                .evaluator(Evaluators.excludeStartPosition());

        return td.traverse(author).nodes();
    }

    public Iterable<Node> getCoAuthorsAndCitators(Integer authorID) {
        Node desiredAuthor = nodeIndex.get(IK_AUTHOR_ID, authorID).getSingle();
        return getCoAuthorsAndCitators(desiredAuthor);
    }

    public Iterable<Node> getCoAuthorsAndCitators(Node author) {
        TraversalDescription td = Traversal.description()
                .evaluator((Evaluators.toDepth(1)))
                .relationships(RelationType.CO_AUTHORSHIP, Direction.OUTGOING)
                .relationships(RelationType.CITATOR, Direction.OUTGOING)
                .evaluator(Evaluators.excludeStartPosition());

        return td.traverse(author).nodes();
    }

    public Node getAuthorByID(int authorID) {
        return nodeIndex.get(IK_AUTHOR_ID, authorID).getSingle();
    }

    private static void registerShutdownHook(final GraphDatabaseService graphDb) {
        // Registers a shutdown hook for the Neo4j instance so that it
        // shuts down nicely when the VM exits (even if you "Ctrl-C" the
        // running example before it's completed)
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                graphDb.shutdown();
            }
        });
    }
}
