package communitiesGraph;

import org.gephi.graph.api.*;
import org.gephi.io.exporter.spi.CharacterExporter;
import org.gephi.io.exporter.spi.GraphExporter;
import org.gephi.project.api.Workspace;
import org.gephi.utils.longtask.spi.LongTask;
import org.gephi.utils.progress.Progress;
import org.gephi.utils.progress.ProgressTicket;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

/**
 * Created with IntelliJ IDEA.
 * User: nikita
 * Date: 26.03.13
 * Time: 12:10
 * To change this template use File | Settings | File Templates.
 */
public class MyExporterCSV implements GraphExporter, CharacterExporter, LongTask {

    protected static final String SEPARATOR = "|";
    protected static final String EOL = "\n";
    //Settings
    protected boolean edgeWeight = true;
    protected boolean writeZero = true;
    protected boolean header = true;
    protected boolean list = false;

    protected boolean normalize = false;
    protected boolean exportColors = true;
    protected boolean exportPosition = true;
    protected boolean exportSize = true;

    //Architecture
    protected Workspace workspace;
    protected Writer writer;
    protected boolean exportVisible;
    protected boolean cancel = false;
    protected ProgressTicket progressTicket;

    //Settings Helper
    private float minSize;
    private float maxSize;
    private float minX;
    private float maxX;
    private float minY;
    private float maxY;
    private float minZ;
    private float maxZ;

    public boolean execute() {
        GraphModel graphModel = workspace.getLookup().lookup(GraphModel.class);
        Graph graph = null;
        if (exportVisible) {
            graph = graphModel.getGraphVisible();
        } else {
            graph = graphModel.getGraph();
        }

        if (normalize) {
            calculateMinMax(graph);
        }
        try {
            exportData(graph);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        return !cancel;
    }

    protected void exportData(Graph graph) throws Exception {
        int max = graph.getNodeCount();

        Progress.start(progressTicket, max);

        if (!list) {
            if (header) {
                writer.append(SEPARATOR);
                Node[] nodes = graph.getNodes().toArray();
                for (int i = 0; i < nodes.length; i++) {
                    writeMatrixNode(nodes[i], i < nodes.length - 1);
                }
                writer.append(EOL);
            }
        }

        if (list) {
            Node[] nodes = graph.getNodes().toArray();
            for (int i = 0; i < nodes.length; i++) {
                Node n = nodes[i];
                List<Node> neighbours = new ArrayList<Node>();
                for (Edge e : graph.getEdges(n)) {
                    if (!e.isDirected() || (e.isDirected() && n == e.getSource())) {
                        Node m = graph.getOpposite(n, e);
                        neighbours.add(m);
                    }
                }

                for (Edge e : ((HierarchicalGraph) graph).getMetaEdges(n)) {
                    if (!e.isDirected() || (e.isDirected() && n == e.getSource())) {
                        Node m = graph.getOpposite(n, e);
                        neighbours.add(m);
                    }
                }

                writeListNode(n, !neighbours.isEmpty());
                for (int j = 0; j < neighbours.size(); j++) {
                    writeListNode(neighbours.get(j), j < neighbours.size() - 1);

                }
                writer.append(EOL);
            }
        } else {
            if (graph instanceof DirectedGraph) {
                DirectedGraph directedGraph = (DirectedGraph) graph;
                Node[] nodes = graph.getNodes().toArray();
                for (Node n : nodes) {
                    if (cancel) {
                        break;
                    }
                    writeMatrixNode(n, true);
                    for (int j = 0; j < nodes.length; j++) {
                        Node m = nodes[j];
                        Edge e = directedGraph.getEdge(n, m);
                        e = e == null ? ((HierarchicalDirectedGraph) directedGraph).getMetaEdge(n, m) : e;
                        writeEdge(e, j < nodes.length - 1);
                    }
                    writer.append(SEPARATOR);
                    writeAllSettings(n);

                    Progress.progress(progressTicket);
                    writer.append(EOL);
                }
            } else if (graph instanceof UndirectedGraph) {
                UndirectedGraph undirectedGraph = (UndirectedGraph) graph;
                Node[] nodes = graph.getNodes().toArray();
                for (Node n : nodes) {
                    if (cancel) {
                        break;
                    }
                    writeMatrixNode(n, true);
                    for (int j = 0; j < nodes.length; j++) {
                        Node m = nodes[j];
                        Edge e = undirectedGraph.getEdge(n, m);
                        e = e == null ? ((HierarchicalUndirectedGraph) undirectedGraph).getMetaEdge(n, m) : e;
                        writeEdge(e, j < nodes.length - 1);
                    }
                    writer.append(SEPARATOR);
                    writeAllSettings(n);

                    Progress.progress(progressTicket);
                    writer.append(EOL);
                }
            } else {
                MixedGraph mixedGraph = (MixedGraph) graph;
                Node[] nodes = graph.getNodes().toArray();
                for (Node n : graph.getNodes()) {
                    if (cancel) {
                        break;
                    }
                    writeMatrixNode(n, true);
                    for (int j = 0; j < nodes.length; j++) {
                        Node m = nodes[j];
                        Edge e = mixedGraph.getEdge(n, m);
                        e = e == null ? ((HierarchicalMixedGraph) mixedGraph).getMetaEdge(n, m) : e;
                        writeEdge(e, j < nodes.length - 1);
                    }
                    writer.append(SEPARATOR);
                    writeAllSettings(n);

                    Progress.progress(progressTicket);
                    writer.append(EOL);
                }
            }
        }

        graph.readUnlockAll();

        Progress.finish(progressTicket);
    }

    protected void writeAllSettings(Node node) throws IOException {
        writeNodePosition(node);
        writer.append(SEPARATOR);
        writeNodeSize(node);
        writer.append(SEPARATOR);
        writeNodeColor(node);
        writer.append(SEPARATOR);
        writeIsCliqueFlag(node);
    }

    protected void writeIsCliqueFlag(Node node) throws IOException {
        Object obj = node.getAttributes().getValue("isClique");
        if (obj != null && ((Boolean) obj)) {
            writer.append("1");
        } else {
            writer.append("0");
        }
    }

    protected void writeNodePosition(Node node) throws IOException {
        float x = node.getNodeData().x();
        if (normalize && x != 0.0) {
            x = (x - minX) / (maxX - minX);
        }
        float y = node.getNodeData().y();
        if (normalize && y != 0.0) {
            y = (y - minY) / (maxY - minY);
        }
        float z = node.getNodeData().z();
        if (normalize && z != 0.0) {
            z = (z - minZ) / (maxZ - minZ);
        }
        if (!(x == 0 && y == 0 && z == 0)) {
            writer.append(String.valueOf(x)).append(SEPARATOR)
                    .append(String.valueOf(y)).append(SEPARATOR)
                    .append(String.valueOf(z));
        }
    }

    protected void writeNodeColor(Node node) throws IOException {
        int r = Math.round(node.getNodeData().r() * 255f);
        int g = Math.round(node.getNodeData().g() * 255f);
        int b = Math.round(node.getNodeData().b() * 255f);
        if (r != 0 || g != 0 || b != 0) {
            writer.append(String.valueOf(r)).append(SEPARATOR)
                    .append(String.valueOf(g)).append(SEPARATOR)
                    .append(String.valueOf(b));
        }
    }

    protected void writeNodeSize(Node node) throws IOException {
        float size = node.getNodeData().getSize();
        if (normalize) {
            size = (size - minSize) / (maxSize - minSize);
        }
        writer.append(String.valueOf(size));
    }

    protected void writeEdge(Edge edge, boolean writeSeparator) throws IOException {
        if (edge != null) {
            if (edgeWeight) {
                writer.append(Float.toString(edge.getWeight()));
            } else {
                writer.append(Float.toString(1f));
            }
            if (writeSeparator) {
                writer.append(SEPARATOR);
            }

        } else {
            if (writeZero) {
                writer.append("0");
            }
            if (writeSeparator) {
                writer.append(SEPARATOR);
            }
        }
    }

    protected void writeMatrixNode(Node node, boolean writeSeparator) throws IOException {
        if (header) {
            String label = node.getNodeData().getLabel();
            if (label == null) {
                label = node.getNodeData().getId();
            }
            writer.append(label);
            if (writeSeparator) {
                writer.append(SEPARATOR);
            }
        }
    }

    protected void writeListNode(Node node, boolean writeSeparator) throws IOException {
        String label = node.getNodeData().getLabel();
        if (label == null) {
            label = node.getNodeData().getId();
        }
        writer.append(label);
        if (writeSeparator) {
            writer.append(SEPARATOR);
        }
    }

    protected void calculateMinMax(Graph graph) {
        minX = Float.POSITIVE_INFINITY;
        maxX = Float.NEGATIVE_INFINITY;
        minY = Float.POSITIVE_INFINITY;
        maxY = Float.NEGATIVE_INFINITY;
        minZ = Float.POSITIVE_INFINITY;
        maxZ = Float.NEGATIVE_INFINITY;
        minSize = Float.POSITIVE_INFINITY;
        maxSize = Float.NEGATIVE_INFINITY;

        for (Node node : graph.getNodes()) {
            NodeData nodeData = node.getNodeData();
            minX = Math.min(minX, nodeData.x());
            maxX = Math.max(maxX, nodeData.x());
            minY = Math.min(minY, nodeData.y());
            maxY = Math.max(maxY, nodeData.y());
            minZ = Math.min(minZ, nodeData.z());
            maxZ = Math.max(maxZ, nodeData.z());
            minSize = Math.min(minSize, nodeData.getSize());
            maxSize = Math.max(maxSize, nodeData.getSize());
        }
    }

    public boolean cancel() {
        cancel = true;
        return true;
    }

    public void setProgressTicket(ProgressTicket progressTicket) {
        this.progressTicket = progressTicket;
    }

    public boolean isEdgeWeight() {
        return edgeWeight;
    }

    public void setEdgeWeight(boolean edgeWeight) {
        this.edgeWeight = edgeWeight;
    }

    public boolean isHeader() {
        return header;
    }

    public void setHeader(boolean header) {
        this.header = header;
    }

    public boolean isWriteZero() {
        return writeZero;
    }

    public void setWriteZero(boolean writeZero) {
        this.writeZero = writeZero;
    }

    public boolean isList() {
        return list;
    }

    public void setList(boolean list) {
        this.list = list;
    }

    public boolean isExportVisible() {
        return exportVisible;
    }

    public void setExportVisible(boolean exportVisible) {
        this.exportVisible = exportVisible;
    }

    public void setWriter(Writer writer) {
        this.writer = writer;
    }

    public Workspace getWorkspace() {
        return workspace;
    }

    public void setWorkspace(Workspace workspace) {
        this.workspace = workspace;
    }

    public boolean isNormalize() {
        return normalize;
    }

    public void setNormalize(boolean normalize) {
        this.normalize = normalize;
    }

    public boolean isExportColors() {
        return exportColors;
    }

    public void setExportColors(boolean exportColors) {
        this.exportColors = exportColors;
    }

    public boolean isExportPosition() {
        return exportPosition;
    }

    public void setExportPosition(boolean exportPosition) {
        this.exportPosition = exportPosition;
    }

    public boolean isExportSize() {
        return exportSize;
    }

    public void setExportSize(boolean exportSize) {
        this.exportSize = exportSize;
    }
}
