﻿//Created by TveMA, 2012
using System;
using System.Collections.Generic;
using System.Collections;
using System.Linq;
using System.Text;
using Irony.Parsing;

namespace TveMA.SearchQueryAnalyser
{
    /// <summary>
    /// kinds of terms used in grammar
    /// </summary>
    enum Term 
    { 
        AdvancedTerm,
        Day,
        Date,
        Ext,
        Extension,
        ExprWithOutTag,
        Full,
        FullExpr,
        GeneralTerm,
        Ident,
        Inequality,     
        LongTag,        
        ManyTags,
        Month,
        MetaExpr,       
        ParseTree,
        PossibleTag,        
        Tag,
        TagBeforeSepr,
        TgS,      
        Year,
        WithExpr,
        WithTerm, 
    }

    /// <summary>
    /// search query grammar
    /// </summary>
    [Language("SearchFileData", "3", "Search of File Data Grammar")]
    class SQGrammar : Grammar
    {
        private HashSet<string> _keyterm = new HashSet<string>(new String[] 
        { "with","about","where","between","at",
             ">","<","=","and"});

        public HashSet<string> KeyTerm
        {
            get { return _keyterm; }
            private set { _keyterm = value; }
        }

        public SQGrammar()
        {
            // Terminals
            IdentifierTerminal identifier = new IdentifierTerminal(GetNameOfTerm(Term.Ident), IdFlags.IsNotKeyword);
            identifier.AllFirstChars += Strings.DecimalDigits;

            IdentifierTerminal ext = new IdentifierTerminal(GetNameOfTerm(Term.Ext), IdFlags.NameIncludesPrefix
                & IdFlags.IsNotKeyword);
            ext.AllChars = Strings.AllLatinLetters;
            ext.AddPrefix(".", IdFlags.NameIncludesPrefix & IdFlags.IsNotKeyword);
            ext.EscapeChar = ' '; ext.AllFirstChars = ".";

            NumberLiteral day = new NumberLiteral(31, GetNameOfTerm(Term.Day), 2, 1);
            day.Flags = NumberFlags.IntOnly; //day.EscapeChar = ' ';  
            NumberLiteral mnth = new NumberLiteral(12, GetNameOfTerm(Term.Month), 1, 2);
            mnth.Flags = NumberFlags.IntOnly;
            NumberLiteral year = new NumberLiteral(GetNameOfTerm(Term.Year), 4, 2);
            year.Flags = NumberFlags.IntOnly;
            KeyTerm WITH = ToTerm("with");
            KeyTerm ABOUT = ToTerm("about");
            KeyTerm WHERE = ToTerm("where");
            KeyTerm comma = ToTerm(",");
            KeyTerm and = ToTerm("and");
            KeyTerm pc = ToTerm(";");
            KeyTerm dot = ToTerm(".");
            KeyTerm greater = ToTerm(">");
            KeyTerm less = ToTerm("<");
            KeyTerm equal = ToTerm("=");
            KeyTerm BETWEEN = ToTerm("between");
            KeyTerm AT = ToTerm("at");

            // Non-terminals 
            var FE = CreateNonTerminal(Term.FullExpr);
            var Tag = CreateNonTerminal(Term.Tag);
            var AdvT = CreateNonTerminal(Term.AdvancedTerm);
            var MTgs = CreateNonTerminal(Term.ManyTags);
            var TgS = CreateNonTerminal(Term.TgS);
            var PTg = CreateNonTerminal(Term.PossibleTag);
            var Ineq = CreateNonTerminal(Term.Inequality);
            var Date = CreateNonTerminal(Term.Date);
            var Sep = new NonTerminal(",");
            Sep.SetOption(TermOptions.IsPunctuation);
            var Extens = CreateNonTerminal(Term.Extension);
            var WithT = CreateNonTerminal(Term.WithTerm);
            var WE = CreateNonTerminal(Term.WithExpr);
            var GT = CreateNonTerminal(Term.GeneralTerm);
            var TBSep = CreateNonTerminal(Term.TagBeforeSepr);
            var F = CreateNonTerminal(Term.Full);
            var SWOTg = CreateNonTerminal(Term.ExprWithOutTag);
            var ME = CreateNonTerminal(Term.MetaExpr);
            var LngTg = CreateNonTerminal(Term.LongTag);

            // BNF rules
            this.Root = F;
            F.Rule = FE
                    | SWOTg + WE;
            FE.Rule = GT + FE
                   | GT
                   | AdvT
                   | AdvT + FE;
            GT.Rule = PTg
                       | Extens;

            WithT.Rule = WITH
                          | ABOUT
                          | WHERE;
            WE.Rule = WithT + PTg
                          | WithT + AdvT
                          | WE + PTg
                          | WE + AdvT;
            WE.SetOption(TermOptions.IsList);

            SWOTg.Rule = GT + SWOTg
                      | GT;

            AdvT.Rule = MTgs
                       | ME;
            MTgs.Rule = TBSep + TgS;
            TgS.Rule = LngTg + TBSep + TgS
                    | LngTg;

            Sep.Rule = pc
                     | comma;
            LngTg.Rule = LngTg + PTg
                      | Tag
                      | Eof;
            TBSep.Rule = Tag + Sep;
            Ineq.Rule = greater
                    | less
                    | equal;
            ME.Rule = Ineq + Date
                          | BETWEEN + Date + Date
                          | AT + Date;
            PTg.Rule = identifier;
            Extens.Rule = ext;
            Tag.Rule = identifier;
            Date.Rule = day + dot + mnth + dot + year
                      | day + dot + mnth;

            // Operators precedence

            MarkTransient(GT, AdvT, LngTg, TgS, SWOTg, FE, Ineq);
        }

        /// <summary>
        /// creates non-terminal by constant from enumeration 
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        private NonTerminal CreateNonTerminal(Term value)
        {
            return new NonTerminal(GetNameOfTerm(value));
        }

        /// <summary>
        /// gets name of constant from enumeration
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        private static string GetNameOfTerm(Term value)
        {
            return Enum.GetName(typeof(Term), value);
        }
    }
}

