﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;

namespace TveMA.SearchQueryAnalyser
{
    /// <summary>
    /// kind of data
    /// </summary>
    public enum DataKind
    {
        Ext,
        Meta,
        NotTag,
        PossibleTag,
        Tag
    }

    public class TreeAnalyser
    {
        private const string _xmlBuffer = "treeBuffer.xml";
        private XmlDocument tree;
        /// <summary>
        /// fills by path of visited and not left terms from xml
        /// </summary>
        private Stack<Term> stack;
        private bool noTags;
        /// <summary>
        /// possible current tag is continuation of last tag
        /// </summary>
        private bool possibleLongTag;
        /// <summary>
        /// how many tags could be one long tag
        /// </summary>
        private int possibleLongTagCount;
        /// <summary>
        /// should to join possible long tags
        /// </summary>
        private bool addPrevToLastTg;
        private Dictionary<DataKind, List<string>> _data;
        /// <summary>
        /// output data
        /// </summary>
        public Dictionary<DataKind, List<string>> Data
        {
            private set { _data = value; }
            get { return _data; }
        }

        private TreeAnalyser() { }

        public TreeAnalyser(XmlDocument inputTree)
        {
            Data = new Dictionary<DataKind, List<string>>();
            foreach (DataKind dK in Enum.GetValues(typeof(DataKind)))
            {
                Data.Add(dK, new List<string>());
            }
            stack = new Stack<Term>();
            tree = inputTree;
        }

        /// <summary>
        /// analyses input tree and fill output data
        /// </summary>
        /// <returns></returns>
        public int AnalyseTree()
        {
            using (FileStream str = File.Create(_xmlBuffer))
            {
                tree.Save(str);
            }
            try
            {
                stack.Push(Term.ParseTree);
                using (XmlTextReader reader = new XmlTextReader(_xmlBuffer))                
                while (reader.Read())
                {
                    switch (reader.NodeType)
                    {
                        case XmlNodeType.Element:
                            if (reader.AttributeCount == 3)
                            {
                                reader.MoveToFirstAttribute();
                                Term kind = (Term)Enum.Parse(typeof(Term), reader.Value);
                                //identifier
                                if (kind == Term.Ident)
                                {
                                    AnalyseIdent(reader);
                                }
                                //extension
                                else if (kind == Term.Ext)
                                {
                                    reader.MoveToAttribute(2);
                                    Data[DataKind.Ext].Add(reader.Value);
                                }
                                else if (kind == Term.Day || kind == Term.Year || kind == Term.Month)
                                {
                                    //to do
                                }
                            }
                            else
                            {
                                if (reader.HasAttributes)
                                    AnalyseTerm(reader);
                            }
                            break;
                        case XmlNodeType.EndElement:
                            stack.Pop();
                            break;
                        default:
                            break;
                    }
                }
            }
            catch { return -1; }
            return 0;
        }

        /// <summary>
        /// analyses situation where the term is current
        /// </summary>
        /// <param name="reader"></param>
        private void AnalyseTerm(XmlTextReader reader)
        {
            reader.MoveToFirstAttribute();
            var termKind = (Term) Enum.Parse(typeof(Term), reader.Value);
            stack.Push(termKind);
            switch (termKind)
            {
                case Term.ExprWithOutTag:
                    noTags = true; 
                    break;
                case Term.WithExpr:
                    noTags = false; 
                    break;
                case Term.TagBeforeSepr:
                    if (possibleLongTag)
                    {
                        possibleLongTag = false;
                        addPrevToLastTg = true;
                    }
                    break;
                case Term.LongTag:
                    possibleLongTag = true; 
                    break;
                case Term.PossibleTag:
                    if (possibleLongTag)
                        possibleLongTagCount++;
                    break;
            }
        }

        /// <summary>
        /// analyses identifier where the term is current
        /// </summary>
        /// <param name="reader"></param>
        private void AnalyseIdent(XmlTextReader reader)
        {
            reader.MoveToAttribute(2);
            switch (stack.Peek())
            {
                case Term.Tag:
                    if (addPrevToLastTg)
                    {
                        addPrevToLastTg = false;
                        for (int i = 0; i < possibleLongTagCount; i++)
                        {
                            int lastIndex1 = Data[DataKind.Tag].Count - 1;
                            int lastIndex2 = Data[DataKind.PossibleTag].Count - 1;
                            Data[DataKind.Tag][lastIndex1] = Data[DataKind.Tag][lastIndex1]
                                + " " + Data[DataKind.PossibleTag][lastIndex2];
                            Data[DataKind.PossibleTag].RemoveAt(lastIndex2);
                        }
                        possibleLongTagCount = 0;
                    }
                    Data[DataKind.Tag].Add(reader.Value);
                    break;
                case Term.PossibleTag:
                    if (noTags)
                        Data[DataKind.NotTag].Add(reader.Value);
                    else
                        Data[DataKind.PossibleTag].Add(reader.Value);
                    break;

            }
        }
    }
}
