﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Data;
using System.Data.SqlClient;
using System.Configuration;
using System.Text;


namespace TveMA.FStorage.Core.Searcher
{
    class Searcher
    {
        public static string ErrorMsg;
        private static int error;
        private static DataTable answer;
        private static string Connect = Core.Properties.Settings.Default.FSConnectionString;

        private Searcher() { }

        /// <summary>
        /// begins search by given seperate data
        /// </summary>
        /// <param name="name">name of file</param>
        /// <param name="ext">extension</param>
        /// <param name="categ">fixed category of file</param>
        /// <param name="tags">string consist of tags</param>
        /// <param name="meta">metadata of file</param>
        /// <returns></returns>
        public static DataTable Search(string name, string ext, string categ, string tags, string meta)
        {
            DataS dS;
            answer = new DataTable(QString.New);
            dS.Name = name;
            dS.Meta = meta;
            dS.Categ = categ;
            dS.Tags = tags.Split(',', ';').ToList();
            dS.Ext = ext;
            if (dS.Suit())
            {
                return SendQuery(dS);
            }
            else
            {
                return new DataTable();
            }
        }

        /// <summary>
        /// searches by sentence 
        /// </summary>
        /// <param name="data">user search query</param>
        /// <returns></returns>
        public static DataTable SearchBySent(string data)
        {
            return SearchBySent(data, "", "", "", "", "");
        }

        /// <summary>
        /// search by sentence
        /// </summary>
        /// <param name="data">not parsed search query</param>
        /// <param name="name">name of file</param>
        /// <param name="ext">extension</param>
        /// <param name="categ">fixed category of file</param>
        /// <param name="tags">string consist of tags</param>
        /// <param name="meta">metadata of file</param>
        /// <returns></returns>
        public static DataTable SearchBySent(string data, string name, string ext, string categ, string tags, string meta)
        {
            DataS dS;
            answer = new DataTable(QString.New);
            dS.Name = name;
            dS.Meta = meta;
            dS.Categ = categ;
            dS.Tags = (String.IsNullOrEmpty(tags)) ? new List<string>() : tags.Split(',', ';').ToList();
            dS.Ext = ext;
            Analyser analyse = new Analyser(dS);
            if (analyse.SetData(data) < 0)
                error = -1;
            if (analyse.SetData(data) > 0)
                analyse.CheckOnCategory(GetCats().Rows[0].ItemArray);
            if (analyse.DataForSearch.Suit())
            {
                return SendQuery(analyse.DataForSearch);
            }
            else
            {
                return new DataTable();
            }
        }

        /// <summary>
        /// sends query to database
        /// </summary>
        /// <param name="dS">keep data for search</param>
        /// <returns></returns>
        private static DataTable SendQuery(DataS dS)
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm = ConstructSqlCmd(cm, dS);
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable(QString.File);
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            answer = result.Copy();
            return result;
        }

        /// <summary>
        /// constructs sql command 
        /// </summary>
        /// <param name="cm">sql command to construct</param>
        /// <param name="dS">keep data for search</param>
        /// <returns></returns>
        private static SqlCommand ConstructSqlCmd(SqlCommand cm, DataS dS)
        {
            bool M = !String.IsNullOrEmpty(dS.Meta);
            bool N = !String.IsNullOrEmpty(dS.Name);
            bool T = dS.Tags.Count != 0;
            bool E = !String.IsNullOrEmpty(dS.Ext);
            bool C = !String.IsNullOrEmpty(dS.Categ);
            if (M | ((E | C) && T))
                cm = BuildSearchQuery(cm, dS);
            else
            {
                cm.CommandText = QString.SELECTAllFr("");
                if (N)
                {
                    cm.Parameters.Add(QParam.Name, SqlDbType.NVarChar, 50);
                    cm.Parameters[0].Value = dS.Name;
                    if (!(T | E | C | M))
                    {
                        cm.CommandText += "dbo.SearchByName(@name)";
                    }
                    else if (T)
                    {
                        cm.CommandText += "dbo.SearchByTagswName(@tags,@name)";
                        cm.Parameters.Add(QParam.Tags, SqlDbType.NText);
                        cm.Parameters[1].Value = dS.Tags;
                    }
                    else if (E)
                    {
                        cm.CommandText += "dbo.SearchByExtwName(@ext,@name)";
                        cm.Parameters.Add(QParam.Ext, SqlDbType.NVarChar, 4);
                        cm.Parameters[1].Value = dS.Ext;
                    }
                    else if (C)
                    {
                        cm.CommandText += "dbo.SearchByCatwName(@cat,@name)";
                        cm.Parameters.Add(QParam.Cat, SqlDbType.NVarChar, 50);
                        cm.Parameters[1].Value = dS.Categ;
                    }
                }
                else if (T)
                {
                    cm.CommandText = QString.SELECTAllFr("dbo.SearchByTags(@tags)");
                    cm.Parameters.Add(QParam.Tags, SqlDbType.NText);
                    cm.Parameters[0].Value = ListOfTagsToString(dS.Tags);
                }
                else if (E)
                {
                    cm.CommandText = QString.SELECTAllFr("dbo.SearchByExt(@ext)");
                    cm.Parameters.Add(QParam.Ext, SqlDbType.NVarChar, 4);
                    cm.Parameters[0].Value = dS.Tags;
                }
                else if (C)
                {
                    cm.CommandText = QString.SELECTAllFr("dbo.SearchByCat(@cat)");
                    cm.Parameters.Add(QParam.Cat, SqlDbType.NVarChar, 50);
                    cm.Parameters[0].Value = dS.Tags;
                }
            }
            return cm;
        }

        /// <summary>
        /// build sql command as more difficult query
        /// </summary>
        /// <param name="cm">sql command to construct</param>
        /// <param name="dS">keep data for search</param>
        /// <returns></returns>
        private static SqlCommand BuildSearchQuery(SqlCommand cm, DataS dS)
        {
            bool M = !String.IsNullOrEmpty(dS.Meta);
            bool N = !String.IsNullOrEmpty(dS.Name);
            bool T = dS.Tags.Count != 0;
            bool E = !String.IsNullOrEmpty(dS.Ext);
            bool C = !String.IsNullOrEmpty(dS.Categ);
            string table = QTable.File;
            table = (N && !E && !C) ? "(" + QString.SELECTAllFr("SearchByName('" + dS.Name + "')") + ")" : table;
            table = (E) ? ((N) ? "(" + QString.SELECTAllFr("SearchByExtwName('" + dS.Ext + "','" + dS.Name + "')") + ")"
                                    : "(" + QString.SELECTAllFr("SearchByExt('" + dS.Ext + "')") + ")")
                            : table;
            table = (C && !E) ? ((N) ? "(" + QString.SELECTAllFr("SearchByCatwName('" + dS.Categ + "','" + dS.Name + "')") + ")"
                                    : "(" + QString.SELECTAllFr("SearchByCat('" + dS.Categ + "')") + ")")
                            : table;
            if (M)
                if (T)
                    table = SbyMeta(table, dS.Meta);
                else
                {
                    cm.CommandText = "(" + QString.SELECTAllFr("SearchByMeta(@meta,@tbl)") + ")";
                    cm.Parameters.Add("@meta", SqlDbType.NVarChar, 50);
                    cm.Parameters.Add("@table", SqlDbType.NVarChar, 50);
                    cm.Parameters[0].Value = dS.Meta;
                    cm.Parameters[1].Value = table;
                    cm.CommandType = CommandType.StoredProcedure;
                    return cm;
                }
            table = SbyTags(table, dS.Tags);
            if (!(M | E | (C && !E) | T) | table == QTable.File) throw new NotImplementedException();
            cm.CommandText = table;
            return cm;
        }

        /// <summary>
        /// build query by list of tags
        /// </summary>
        /// <param name="table">name of table or sql function to get data from</param>
        /// <param name="data">list of tags</param>
        /// <returns></returns>
        private static string SbyTags(string table, List<string> data)
        {
            string tags = ListOfTagsToString(data);
            string tagNames = QString.SELECTAllFr("Charlist_to_table('" + tags + "', DEFAULT)");
            string filesId = QString.SELFrWh("Ft.file_id, count(*) cnt "
                                            , QTable.TagOfFile + " Ft," + QTable.Tag + " T "
                                            , "Ft.tag_id = T.id AND name " + "IN (" + tagNames + ")" + "GROUP BY file_id)");
            return QString.SELFrWh(QString.FullDataList
                                    , table + " TBL, (" + filesId + ") F "
                                    , "TBL.id = F.file_id ORDER BY cnt");
        }

        /// <summary>
        /// makes string contained tags from list seperated by separator
        /// </summary>
        /// <param name="data">list of tags</param>
        /// <returns></returns>
        private static string ListOfTagsToString(List<string> data)
        {
            string tags = "";
            foreach (string t in data)
                tags += t + ",";
            tags = tags.Remove(tags.Length - 1);
            return tags;
        }

        /// <summary>
        /// search by metadata
        /// </summary>
        /// <param name="table">name of table or sql function to get data from</param>
        /// <param name="data">metadata in string with suitable format</param>
        /// <returns></returns>
        private static string SbyMeta(string table, string data)
        {
            return QString.SELECTAllFr(table) + QString.WHERE("") + data;
        }

        /// <summary>
        /// search by category of file
        /// </summary>
        /// <param name="table">name of table or sql function to get data from</param>
        /// <param name="data">name of category</param>
        /// <returns></returns>
        private static string SbyCat(string table, string data)
        {
            return QString.SELECTAllFr(table) + QString.WHERE("id_ext") + " IN (" +
                               QString.SELFr("id", "GetExtOfCat(" + data + ")") + ")";
        }

        /// <summary>
        /// search by extension
        /// </summary>
        /// <param name="table">name of table or sql function to get data from</param>
        /// <param name="data">extension of searched file</param>
        /// <returns></returns>
        private static string SbyExt(string table, string data)
        {
            return QString.SELECTAllFr(table) + QString.WHERE("id_ext = ") + data;
        }

        /// <summary>
        /// get all categories of files
        /// </summary>
        /// <returns>table with list of categories</returns>
        internal static DataTable GetCats()
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm.CommandText = QString.SELECTAllFr("dbo.GetCategs()");
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable("Category");
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            return result;
        }

        /// <summary>
        /// gets all extensions
        /// </summary>
        /// <returns></returns>
        internal static DataTable GetExt()
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm.CommandText = QString.SELECTAllFr("dbo.GetExt()");
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable("Extension");
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            return result;
        }

        /// <summary>
        /// gets extensions of given category
        /// </summary>
        /// <param name="categName">name of category</param>
        /// <returns></returns>
        internal static DataTable GetExt(string categName)
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm.CommandText = QString.SELECTAllFr("dbo.GetExtOfCat(@cat)");
                cm.Parameters.Add(QParam.Cat, SqlDbType.NVarChar, 20);
                cm.Parameters[0].Value = categName;
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable("Extensions of " + categName);
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            return result;
        }

        /// <summary>
        /// get all files from storage
        /// </summary>
        /// <returns>table with all file's data</returns>
        internal static DataTable GetAll()
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm.CommandText = QString.SELFr(QString.FullDataList, QTable.File);
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable(QString.File);
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            answer = result.Copy();
            return result;
        }

        /// <summary>
        /// get only opened files which havent been saved
        /// </summary>
        /// <returns>table with opened files</returns>
        internal static DataTable GetOpenedFiles()
        {
            DataTable result = new DataTable();
            SqlConnection cn = null;
            try
            {
                cn = new SqlConnection(Connect);
                SqlCommand cm = cn.CreateCommand();
                cm.CommandText = QString.SELFr("file_id, name", QTable.OpF);
                cn.Open();
                SqlDataReader reader = cm.ExecuteReader(CommandBehavior.CloseConnection);
                result = new DataTable(QTable.OpF);
                result.Load(reader);
            }
            catch (SqlException ex)
            {
                ErrorMsg = ex.Message;
                error = -1;
                result = new DataTable(QString.Null);
            }
            finally
            {
                if (cn != null)
                {
                    cn.Close();
                }
            }
            answer = result.Copy();
            return result;
        }
    }
}
