﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Irony.Parsing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Xml;

namespace TveMA.SearchQueryAnalyser
{
    /// <summary>
    /// search query parser
    /// </summary>
    public class SQParser
    {
        Grammar grammar;
        LanguageData language;
        Parser parser;
        ParsingContext parsingContext;
        ParseTree parseTree;
        string _elapsedTime;
        string _source;
        private TreeView tvParseTree;
        private List<CompErrors> gridCompileErrors;

        public SQParser(string source)
        {
            tvParseTree = new TreeView();
            gridCompileErrors = new List<CompErrors>();

            grammar = getGrammar();
            _source = source;
            ConstructParser();
        }

        /// <summary>
        /// initializes a new instance of grammar
        /// </summary>
        /// <returns></returns>
        private Grammar getGrammar()
        {
            Grammar grammar = new SQGrammar();
            return grammar; 
        }

        /// <summary>
        /// constructs parser
        /// </summary>
        private void ConstructParser()
        {
            parseTree = null;
            grammar.ParseMethod = ParseMethod.Lalr;
            Stopwatch sw = new Stopwatch();
            try
            {
                sw.Start();
                language = new LanguageData(grammar);
                parser = new Parser(language);
                parsingContext = new ParsingContext(parser);
                sw.Stop();
            }
            finally
            {
                _elapsedTime = sw.ElapsedMilliseconds.ToString();
            }
        }

        /// <summary>
        /// parses source
        /// </summary>
        public void Parse()
        {
            ParseSample();           
        }

        /// <summary>
        /// returns
        /// </summary>
        /// <returns></returns>
        public TreeView GetTree()
        {
            ShowParseTree();
            return tvParseTree;
        }

        public List<CompErrors> GetErrors()
        {
            ShowCompilerErrors();
            return gridCompileErrors;
        }
        public XmlDocument GetXmlTree()
        {            
            return parseTree.ToXmlDocument();
        }

        private void ParseSample()
        {
            if (parser == null || !parser.Language.CanParse()) return;
            parseTree = null;
            parsingContext = new ParsingContext(parser);
            parsingContext.SetOption(ParseOptions.TraceParser, true);
            try
            { 
                if(String.IsNullOrEmpty(_source))
                    throw new Exception("there is no any input");
                parser.Parse(parsingContext, _source, "<source>");
            }
            catch (Exception ex)
            {
                gridCompileErrors.Add(new CompErrors(null, ex.Message, null));
                throw;
            }
            finally
            {
                parseTree = parsingContext.CurrentParseTree;
                ShowCompilerErrors();
                ShowParseTree();
            }
        }

        private void ShowCompilerErrors()
        {
            gridCompileErrors.Clear();
            if (parseTree == null || parseTree.Errors.Count == 0) return;
            foreach (var err in parseTree.Errors)
                gridCompileErrors.Add(new CompErrors(err.Location, err.Message, err.ParserState));
        }

        private void ShowParseTree()
        {
            tvParseTree.Nodes.Clear();
            if (parseTree == null) return;
            AddParseNodeRec(null, parseTree.Root);
        }

        private void AddParseNodeRec(TreeNode parent, ParseTreeNode nodeInfo)
        {
            if (nodeInfo == null) return;
            string txt = nodeInfo.ToString();
            TreeNode newNode = (parent == null ?
              tvParseTree.Nodes.Add(txt) : parent.Nodes.Add(txt));
            newNode.Tag = nodeInfo;
            foreach (var child in nodeInfo.ChildNodes)
                AddParseNodeRec(newNode, child);
        }
    }

    public struct CompErrors
    {
        public CompErrors(Nullable<SourceLocation> loc, string err, ParserState state)
        {
            this.loc = loc; this.err = err; this.state = state;
        }
        Nullable<SourceLocation> loc;
        string err;
        ParserState state; 
    }
}
