using System;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Configuration;
using System.Security;
using System.ComponentModel;
using System.Reflection;
using System.Text.RegularExpressions;

namespace Utils.Log.Diagnostics
{
  /// <summary>
  /// The trace listener is used to split trace files and write trace files by cycle.
  /// The class uses XmlWriterTraceListener as a base class to provide message output
  /// in corresponding format.
  /// </summary>
  public class CyclicTraceListener : XmlWriterTraceListener
  {
    #region Constants strings
    /// <summary>
    /// File pattern parameter name for Domain parameter.
    /// </summary>
    private const string parameterDomain = "\\(domain\\)";

    /// <summary>
    /// This string is used for Domain parameter in working file pattern.
    /// </summary>
    private const string formatParameterDomain = "{1}";

    /// <summary>
    /// File pattern parameter name for sequence number parameter.
    /// </summary>
    private const string parameterNumber = "\\(number\\)";

    /// <summary>
    /// This string is used for sequence number parameter in working file pattern.
    /// </summary>
    private const string formatParameterNumber = "{{0:D{0}}}";

    /// <summary>
    /// Current namespace used to initialize CyclicTraceListener and write
    /// events using it as source.
    /// </summary>
    private static string namespaceDiagnostics =
      typeof(CyclicTraceListener).Namespace;

    /// <summary>
    /// This message is used when initialization fails.
    /// </summary>
    private const string msgCantInit = "Error occurred. Message output has been disabled. Error:\n{0}";

    /// <summary>
    /// This message is used when the listener can't open target file.
    /// </summary>
    private const string msgCantOpen = "Couldn't open file '{0}'. Trying another sequence numbers. Error:\n{1}";

    /// <summary>
    /// This message is used when the listener can't open any of allowed files.
    /// </summary>
    private const string msgCantRedirect = "Couldn't open any allowed trace file. Message output has been disabled.";

    /// <summary>
    /// This message is used when an initialization parameter is out of range.
    /// </summary>
    private const string msgErrParam = "Parameter '{0}' value '{1}' is out of range. Used default value '{2}'.";
    #endregion

    #region private members
    /// <summary>
    /// Contains absolute path to the folder current application
    /// was started from.
    /// </summary>
    private static readonly string applicationFolder =
      AppDomain.CurrentDomain.BaseDirectory;

    /// <summary>
    /// Contains absolute path to the temporary folder of local user.
    /// </summary>
    private static readonly string tempFolder =
      Path.GetTempPath();

    /// <summary>
    /// Filename pattern for getting target filenames.
    /// </summary>
    private string filePattern;

    /// <summary>
    /// Current number of file in file sequence.
    /// </summary>
    private long fileSeqNum;

    /// <summary>
    /// Current number of written messages.
    /// </summary>
    private long numMessages;

    /// <summary>
    /// Maximum number of files in file sequence.
    /// </summary>
    private long maxFiles;

    /// <summary>
    /// Maximum number of messages per file.
    /// </summary>
    private long maxMessages;

    /// <summary>
    /// Indicates if listener should use temporary or application folder.
    /// </summary>
    private bool useTempFolder;

    /// <summary>
    /// Flag showing if writing messages is blocked.
    /// </summary>
    private bool outputBlocked;

    /// <summary>
    /// Locks request to ConfigurationManager.
    /// Fixes not thread safe custom section creation (MS bug!).
    /// </summary>
    private static object getSectionLocker = new object();

    #endregion

    #region public methods

    /// <summary>
    /// Used to configure TraceSource with configuration file.
    /// </summary>
    /// <param name="fileName">Pattern filename</param>
    public CyclicTraceListener(string fileName)
      : base(TextWriter.Null)
    {
      CyclicTraceConfig traceConfig = null;
      // Fixes not thread safe custom section creation (MS bug!).
      lock(getSectionLocker)
      {
        traceConfig =
          ConfigurationManager.GetSection(namespaceDiagnostics) as CyclicTraceConfig;
      }

      // We should not fail with null reference exception.
      if (traceConfig == null)
      {
        BlockMessageOutput(
          "Could not find configuration section for namespace {0} in application configuration.",
          namespaceDiagnostics);

        return;
      }

      ApplyConfiguration(fileName, traceConfig);
    }

    #region Overriden methods
    /// <summary>
    /// All trace methods used by TraceSource are overridden to control output files.
    /// Each such method at first calls CheckOutputFile method to check if we should change output
    /// file or don't write message at all.
    /// </summary>
    public override void TraceEvent(TraceEventCache eventCache, string source, TraceEventType eventType, int id)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceEvent(eventCache, source, eventType, id);
    }

    /// <summary>
    /// Traces an event.
    /// </summary>
    public override void TraceEvent(
      TraceEventCache eventCache,
      string source,
      TraceEventType eventType,
      int id,
      string format,
      params object[] args)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceEvent(eventCache, source, eventType, id, format, args);
    }

    /// <summary>
    /// Traces an event.
    /// </summary>
    public override void TraceEvent(
      TraceEventCache eventCache,
      string source,
      TraceEventType eventType,
      int id,
      string message)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceEvent(eventCache, source, eventType, id, message);
    }

    /// <summary>
    /// Traces an event.
    /// </summary>
    public override void TraceTransfer(
      TraceEventCache eventCache,
      string source,
      int id,
      string message,
      Guid relatedActivityId)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceTransfer(eventCache, source, id, message, relatedActivityId);
    }

    /// <summary>
    /// Traces an event.
    /// </summary>
    public override void TraceData(
      TraceEventCache eventCache,
      string source,
      TraceEventType eventType,
      int id,
      object data)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceData(eventCache, source, eventType, id, data);
    }

    /// <summary>
    /// Traces an event.
    /// </summary>
    public override void TraceData(
      TraceEventCache eventCache,
      string source,
      TraceEventType eventType,
      int id,
      params object[] data)
    {
      if (!CheckOutputFile())
      {
        return;
      }
      base.TraceData(eventCache, source, eventType, id, data);
    }
    #endregion

    #endregion

    #region private methods
    /// <summary>
    /// Initializes CyclicTraceListener instance and applies configuration properties.
    /// </summary>
    private void ApplyConfiguration(string fileName, CyclicTraceConfig configuration)
    {
      try
      {
        // This is a first file in a sequence.
        this.fileSeqNum = 0;

        // Output is not blocked.
        outputBlocked = false;

        // Checking maxFiles parameter range
        this.maxFiles =
          InitParameter(configuration.MaxFiles, CyclicTraceConfig.MinMaxFiles, CyclicTraceConfig.MaxMaxFiles,
                        CyclicTraceConfig.DefaultMaxFiles, CyclicTraceConfig.NameMaxFiles);

        // Checking maxMessages parameter range
        this.maxMessages =
          InitParameter(configuration.MaxMessages, CyclicTraceConfig.MinMaxMessages, CyclicTraceConfig.MaxMaxMessages,
                        CyclicTraceConfig.DefaultMaxMessages, CyclicTraceConfig.NameMaxMessages);

        // Fortunately, there is nothing to validate here ;)
        // Just read value of useTempFolder parameter from configuration
        this.useTempFolder = configuration.UseTempFolder;

        // Initializing first trace file
        InitFile(fileName);
      }
      catch (FormatException e)
      {
        BlockMessageOutput(msgCantInit, e);
      }
      catch (SecurityException e)
      {
        BlockMessageOutput(msgCantInit, e);
      }
      catch (IOException e)
      {
        BlockMessageOutput(msgCantInit, e);
      }
      catch (NotSupportedException e)
      {
        BlockMessageOutput(msgCantInit, e);
      }
      catch (SystemException e)
      {
        BlockMessageOutput(msgCantInit, e);
      }
    }

    /// <summary>
    /// Initializes configuration parameter (maxFiles and maxMessages).
    /// In case of error it writes an error into Event Log.
    /// </summary>
    /// <param name="value">Parameter value</param>
    /// <param name="minValue">Minimum value</param>
    /// <param name="maxValue">Maximum value</param>
    /// <param name="defaultValue">Default value</param>
    /// <param name="name">Parameter name</param>
    /// <returns>Parameter value which should be used</returns>
    private static long InitParameter(long value, long minValue, long maxValue, long defaultValue, string name)
    {
      if (value < minValue || value > maxValue)
      {
        WriteErrorToEventLog(
          string.Format(
            CultureInfo.InvariantCulture,
            msgErrParam,
            name,
            value,
            defaultValue
            )
          );

        return defaultValue;
      }
      else
      {
        return value;
      }
    }

    /// <summary>
    /// Initializes first trace file.
    /// </summary>
    /// <param name="fileName">Target filename used as pattern for trace files.</param>
    private void InitFile(string fileName)
    {
      // Building trace file pattern
      filePattern = CreatePattern(fileName, maxFiles);
      string curFileName = String.Empty;
      string directoryName = String.Empty;

      // Try to select first file name and initialize sequence number.
      curFileName = SelectFirstFileName();

      // Ensure logs directory exists
      directoryName = Path.GetDirectoryName(curFileName);
      Directory.CreateDirectory(directoryName);

      ChangeOutputFile();
    }

    /// <summary>
    /// Creates file name pattern from the given file name.
    /// </summary>
    private static string CreatePattern(string fileName, long maxFiles)
    {
      int lenSeqNum = maxFiles.ToString(CultureInfo.InvariantCulture).Length;
      string resultFormatParameterNumber = String.Format(
        CultureInfo.InvariantCulture,
        formatParameterNumber,
        lenSeqNum);

      string result = Regex.Replace(fileName, parameterDomain, formatParameterDomain, RegexOptions.IgnoreCase);

      return Regex.Replace(result, parameterNumber, resultFormatParameterNumber, RegexOptions.IgnoreCase);
    }

    /// <summary>
    /// Selects first file name and modifies sequence number.
    /// </summary>
    private string SelectFirstFileName()
    {
      DateTime curDate = DateTime.MaxValue;
      string curFileName = null;

      // Defining Sequence number
      for (long i = 1; i <= maxFiles; ++i)
      {
        curFileName = BuildFileName(filePattern, i, useTempFolder);

        // Check if we found a hole in numeration
        if (!File.Exists(curFileName))
        {
          fileSeqNum = i - 1;
          break;
        }

        // Check oldest file
        DateTime curFileDate = File.GetLastWriteTime(curFileName);
        if (curFileDate < curDate)
        {
          curDate = curFileDate;
          fileSeqNum = i - 1;
        }
      }
      return curFileName;
    }

    /// <summary>
    /// Writes error message into Event Log and blocks message output.
    /// </summary>
    /// <param name="msgErr">Error message.</param>
    /// <param name="msgInfo">Parameter for error message.</param>
    private void BlockMessageOutput(string msgErr, object msgInfo)
    {
      string msg = string.Format(CultureInfo.InvariantCulture, msgErr, msgInfo);
      BlockMessageOutput(msg);
    }

    /// <summary>
    /// Writes error message into Event Log and blocks message output.
    /// </summary>
    /// <param name="msg">Error message.</param>
    private void BlockMessageOutput(string msg)
    {
      WriteErrorToEventLog(msg);
      outputBlocked = true;
    }

    /// <summary>
    /// Builds trace filename by pattern filename and by given sequence number.
    /// </summary>
    /// <param name="pattern">File name pattern to use.</param>
    /// <param name="seqNum">Trace file sequence number.</param>
    /// <param name="useTempFolder">
    /// Should file be placed in temporary folder or in application folder.
    /// </param>
    /// <returns>Trace filename.</returns>
    private static string BuildFileName(string pattern, long seqNum, bool useTempFolder)
    {
      string dllName = AppDomain.CurrentDomain.FriendlyName.Replace(applicationFolder, string.Empty);
      dllName = dllName.Replace(":", string.Empty);
      dllName = dllName.Replace("\\", string.Empty);
      dllName = dllName.Replace(" ", string.Empty);
      return Path.Combine(
        useTempFolder ? tempFolder : applicationFolder,
        String.Format(CultureInfo.InvariantCulture,
          pattern, seqNum, dllName)
        );
    }

    /// <summary>
    /// Generates next trace file sequence number.
    /// </summary>
    private void GenerateNextSequenceNumber()
    {
      fileSeqNum = (fileSeqNum % maxFiles) + 1;
    }

    /// <summary>
    /// Write an error with a text given to event log.
    /// </summary>
    /// <param name="stringToWrite">Error text.</param>
    private static void WriteErrorToEventLog(string stringToWrite)
    {
      try
      {
        EventLog.WriteEntry(namespaceDiagnostics, stringToWrite, EventLogEntryType.Error);
      }
      // If any exception occurs, we just ignore it and work without diagnostics.
      catch (InvalidOperationException)
      {
        // just ignore - nothing to do with it.
      }
      catch (Win32Exception)
      {
        // just ignore - nothing to do with it.
      }
    }

    /// <summary>
    /// Redirects trace output to new file.
    /// </summary>
    private void ChangeOutputFile()
    {
      if (this.Writer != TextWriter.Null)
      {
        this.Writer.Close();
      }

      if (!ChooseNewFile())
      {
        BlockMessageOutput(msgCantRedirect);
      }
    }

    /// <summary>
    /// Tries to choose new file for writing log entries. If new file found,
    /// returns true. Otherwise returns false, which means that this trace source
    /// is not functional any more. 
    /// </summary>
    private bool ChooseNewFile()
    {

      // Trying to open another file
      for (long i = 0; i < maxFiles; ++i)
      {
        string errMessage = null;
        string fileName = string.Empty;

        try
        {
          GenerateNextSequenceNumber();
          fileName = BuildFileName(filePattern, fileSeqNum, useTempFolder);
          this.Writer = new StreamWriter(fileName);
          return true;
        }
        catch (IOException e)
        {
          errMessage = e.Message;
        }
        catch (SecurityException e)
        {
          errMessage = e.Message;
        }
        catch (Win32Exception e)
        {
          errMessage = e.Message;
        }

        if ((0 == i) && !String.IsNullOrEmpty(errMessage))
        {
          WriteErrorToEventLog(
            string.Format(
              CultureInfo.InvariantCulture,
              msgCantOpen,
              fileName,
              errMessage
              )
            );
        }
      }

      return false;
    }

    /// <summary>
    /// The method counts messages in trace files. If limit number of messages is exceeded,
    /// it redirects trace output to another file.
    /// </summary>
    /// <returns>False if current message should be missed.</returns>
    private bool CheckOutputFile()
    {
      while (!outputBlocked)
      {
        ++numMessages;
        if (numMessages <= maxMessages)
        {
          return true;
        }

        // Redirecting trace to a new file
        ChangeOutputFile();
        numMessages = 0;
      }

      return false;
    }
    #endregion
  }//end CyclicTraceListener
}
