using System.Configuration;


namespace Utils.Log.Diagnostics
{
  /// <summary>
  /// The class implements custom configuration section for the Diagnostics component.
  /// The section has 2 attributes "maxFiles" and "maxMessages".
  /// </summary>
  public class CyclicTraceConfig : ConfigurationSection
  {
    #region Constant values
    /// <summary>
    /// Minimum allowed files number
    /// </summary>
    public const long MinMaxFiles = 1;
    /// <summary>
    /// Maximum allowed files number
    /// </summary>
    public const long MaxMaxFiles = 10000;
    /// <summary>
    /// Minimum allowed messages number
    /// </summary>
    public const long MinMaxMessages = 1;
    /// <summary>
    /// Maximum allowed messages number
    /// </summary>
    public const long MaxMaxMessages = 1000000;
    /// <summary>
    /// Default files number
    /// </summary>
    public const long DefaultMaxFiles = 100;
    /// <summary>
    /// Default messages number
    /// </summary>
    public const long DefaultMaxMessages = 1000;
    /// <summary>
    /// Config property name for maxMessages
    /// </summary>
    public const string NameMaxMessages = "maxMessages";
    /// <summary>
    /// Config property name for maxFiles
    /// </summary>
    public const string NameMaxFiles = "maxFiles";
    /// <summary>
    /// Config property name for useTempFolder
    /// </summary>
    public const string NameUseTempFolder = "useTempFolder";
    #endregion

    #region Private vars
    private static ConfigurationPropertyCollection properties;

    /// <summary>
    /// The maxMessages configuration property.
    /// </summary>
    private static readonly ConfigurationProperty maxMessages =
        new ConfigurationProperty(NameMaxMessages,
        typeof(long), DefaultMaxMessages,
        ConfigurationPropertyOptions.IsRequired);

    /// <summary>
    /// The maxFiles configuration property.
    /// </summary>
    private static readonly ConfigurationProperty maxFiles =
        new ConfigurationProperty(NameMaxFiles,
        typeof(long), DefaultMaxFiles,
        ConfigurationPropertyOptions.IsRequired);

    /// <summary>
    /// The useTempFolder configuration property. If set to true,
    /// listener should store log files into temporary folder of current
    /// user. If set to false - log files are stored in the folder, application
    /// is started from. Default value is false.
    /// </summary>
    private static readonly ConfigurationProperty useTempFolder =
        new ConfigurationProperty(NameUseTempFolder, typeof(bool), false);
    #endregion

    #region Constructor
    /// <summary>
    /// CyclicTraceConfig constructor.
    /// </summary>
    public CyclicTraceConfig()
    {
      // Property initialization
      properties = new ConfigurationPropertyCollection();
      properties.Add(maxMessages);
      properties.Add(maxFiles);
      properties.Add(useTempFolder);
    }
    #endregion

    #region Protected properties
    /// <summary>
    /// This is a key customization. It returns the initialized property bag.
    /// </summary>
    protected override ConfigurationPropertyCollection Properties
    {
      get { return properties; }
    }
    #endregion

    #region Public properties
    /// <summary>
    /// Maximum messages per file
    /// </summary>
    [LongValidator(MinValue = MinMaxMessages, MaxValue = MaxMaxMessages, ExcludeRange = false)]
    public long MaxMessages
    {
      get { return (long)this[maxMessages.Name]; }
      set { this[maxMessages.Name] = value; }
    }

    /// <summary>
    /// Maximum trace files
    /// </summary>
    [LongValidator(MinValue = MinMaxFiles, MaxValue = MaxMaxFiles, ExcludeRange = false)]
    public long MaxFiles
    {
      get { return (long)this[maxFiles.Name]; }
      set { this[maxFiles.Name] = value; }
    }

    /// <summary>
    /// If set to true, listener should store log files into temporary folder of 
    /// current user. If set to false - log files are stored in the folder, application
    /// is started from. Default value is false.
    /// </summary>
    public bool UseTempFolder
    {
      get { return (bool)this[useTempFolder.Name]; }
      set { this[useTempFolder.Name] = value; }
    }
    #endregion
  }//end CyclicTraceConfig
}
