using System;
using System.Collections;

namespace HarryBonesSolutions.Performance
{
	/// <summary>
	/// A collection of TimedMethod objects.
	/// </summary>
	public class TimedMethodCollection : CollectionBase
	{
		/// <summary>
		/// Initializes a new instance of the TimedMethod class.
		/// </summary>
		public TimedMethodCollection() : base(){}

		/// <summary>
		/// Adds a TimedMethod object to the collection.
		/// </summary>
		/// <param name="timedMethod">The TimedMethod to be added.</param>
		/// <returns>The TimedMethod instance added.</returns>
		public TimedMethod Add(TimedMethod timedMethod)
		{
			List.Add(timedMethod);
			return timedMethod;
		}

		/// <summary>
		/// Gets or sets the TimedMethod instance at the specified index.
		/// </summary>
		public TimedMethod this[int index]
		{
			get{ return (TimedMethod)List[index]; }
			set{ List[index] = value; }
		}

		/// <summary>
		/// Removes the TimedMethod instance at the specified index.
		/// </summary>
		/// <param name="timedMethod"></param>
		public void Remove(TimedMethod timedMethod)
		{
			List.Remove(timedMethod);
		}

		/// <summary>
		/// Retrieves an array of delegate objects representing the
		///  Method property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.
		/// </summary>
		/// <returns>An array of delegate objects representing the
		///  Method property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.</returns>
		public Delegate[] GetMethods()
		{
			Delegate[] dels = new Delegate[Count];

			for(int i = 0; i < dels.Length; i++)
			{
				dels[i] = this[i].Method;
			}

			return dels;
		}

		/// <summary>
		/// Retrieves a jagged array of objects representing the 
		/// arguments to be passed to the main method for each 
		/// <see cref="TimedMethod"/> in the collection.
		/// </summary>
		/// <returns>A jagged array of objects representing the 
		/// arguments to be passed to the main method for each 
		/// <see cref="TimedMethod"/> in the collection.</returns>
		public object[][] GetArgs()
		{
			object[][] args = new object[Count][];
			
			for(int i = 0; i < args.Length; i++)
			{
				args[i] = this[i].Args;
			}

			return args;
		}

		/// <summary>
		/// Retrieves an array of <c>int</c>s representing the number of 
		/// times the method being timed should be executed.
		/// </summary>
		/// <returns>An array of <c>int</c>s representing the number of 
		/// times the method being timed should be executed.</returns>
		public int[] GetIterations()
		{
			int[] iters = new int[Count];

			for(int i = 0; i < iters.Length; i++)
			{
				iters[i] = this[i].Iterations;
			}

			return iters;
		}

		/// <summary>
		/// Retrieves an array of string representing the display names
		///  for each <see cref="TimedMethod"/> object in the collection.
		/// </summary>
		/// <returns>An array of string representing the display names
		///  for each <see cref="TimedMethod"/> object in the collection.</returns>
		public string[] GetNames()
		{
			string[] names = new string[Count];

			for(int i = 0; i < names.Length; i++)
			{
				names[i] = this[i].DisplayName;
			}

			return names;
		}

		/// <summary>
		/// Retrieves an array of delegate objects representing the
		///  SetUp property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.
		/// </summary>
		/// <returns>An array of delegate objects representing the
		///  SetUp property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.</returns>
		public Delegate[] GetSetUps()
		{
			Delegate[] dels = new Delegate[Count];

			for(int i = 0; i < dels.Length; i++)
			{
				dels[i] = this[i].SetUp;
			}

			return dels;
		}

		/// <summary>
		/// Retrieves a jagged array of objects representing the 
		/// arguments to be passed to the set-up method for each 
		/// <see cref="TimedMethod"/> in the collection.
		/// </summary>
		/// <returns>A jagged array of objects representing the 
		/// arguments to be passed to the set-up method for each 
		/// <see cref="TimedMethod"/> in the collection.</returns>
		public object[][] GetSetUpArgs()
		{
			object[][] args = new object[Count][];
			
			for(int i = 0; i < args.Length; i++)
			{
				args[i] = this[i].SetUpArgs;
			}

			return args;
		}

		/// <summary>
		/// Retrieves an array of delegate objects representing the
		///  Method property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.
		/// </summary>
		/// <returns>An array of delegate objects representing the
		///  Method property of the <see cref="TimedMethod"/>
		///  objects contained in the collection.</returns>
		public Delegate[] GetTearDowns()
		{
			Delegate[] dels = new Delegate[Count];

			for(int i = 0; i < dels.Length; i++)
			{
				dels[i] = this[i].TearDown;
			}

			return dels;
		}

		/// <summary>
		/// Retrieves a jagged array of objects representing the 
		/// arguments to be passed to the tear-down method for each 
		/// <see cref="TimedMethod"/> in the collection.
		/// </summary>
		/// <returns>A jagged array of objects representing the 
		/// arguments to be passed to the tear-down method for each 
		/// <see cref="TimedMethod"/> in the collection.</returns>
		public object[][] GetTearDownArgs()
		{
			object[][] args = new object[Count][];
			
			for(int i = 0; i < args.Length; i++)
			{
				args[i] = this[i].TearDownArgs;
			}

			return args;
		}
	}
}