using System;
using System.IO;
using System.Text;

namespace HarryBonesSolutions.Performance
{
	/// <summary>
	/// This class simplifies dealing with several <see cref="ExecutionTimer"/> 
	/// objects at once by offering methods to change properties on all
	/// objects being managed with one call.
	/// </summary>
	public class ExecutionTimerManager
	{
		private ExecutionTimerCollection _timers;
		private PrintString _beginPrintString;
		private PrintString _endPrintString;
		private PrintString _seriesBeginPrintString;
		private PrintString _seriesEndPrintString;
		private PrintString _collectionBeginPrintString;
		private PrintString _collectionEndPrintString;
		private TextWriter  _out;
		private string _beginDelim;
		private string _endDelim;
		private string _seriesBeginDelim;
		private string _seriesEndDelim;
		private string _collectionBeginDelim;
		private string _collectionEndDelim;
		private bool   _setOutOnWrite;

		/// <summary>
		/// Instantiates a new instance of the <see cref="ExecutionTimerManager"/> 
		/// class.
		/// </summary>
		public ExecutionTimerManager()
		{
			_out = Console.Out;
			_setOutOnWrite = true;
			_timers = new ExecutionTimerCollection();
			_beginPrintString = null;
			_endPrintString = null;
			_seriesBeginPrintString = null;
			_seriesEndPrintString = null;
			_beginDelim = string.Empty;
			_endDelim = string.Empty;
			_seriesBeginDelim = string.Empty;
			_seriesEndDelim = string.Empty;
		}

		/// <summary>
		/// Indexer for the <see cref="ExecutionTimerManager"/> class.
		/// </summary>
		public ExecutionTimer this[string name]
		{
			get
			{
				return _timers[name];
			}
		}

		/// <summary>
		/// Clears the collection of <see cref="ExecutionTimer"/> objects
		///  currently being managed.
		/// </summary>
		public void ClearTimers()
		{
			_timers.Clear();
		}

		/// <summary>
		/// Adds an <see cref="ExecutionTimer"/> object to the collection.
		/// </summary>
		/// <param name="timer"></param>
		public void AddTimer(ExecutionTimer timer)
		{
			// Validate arg

			if(timer == null)
				throw new ArgumentNullException("timer", "Cannot manage a null ExecutionTimer");

			if(_timers.ContainsName(timer.Name))
				throw new NameExistsException("An ExecutionTimer with the same name is already being managed.", timer.Name);

			_timers.Add(timer);
		}

		/// <summary>
		/// Gets a value indicating whether an <see cref="ExecutionTimer"/> 
		/// object with the specified name is already being managed.
		/// </summary>
		/// <param name="name"></param>
		/// <returns></returns>
		public bool NameTaken(string name)
		{
			return _timers.ContainsName(name);
		}

		/// <summary>
		/// Gets or sets the <see cref="TextWriter"/> used when writing results.
		/// </summary>
		/// <remarks>When set, this property will set the Out property of each
		///  <see cref="ExecutionTimer"/> object being managed to the specified value.  The 
		///  Out property of each <see cref="ExecutionTimer"/> being managed will be set to
		///   the specified value when WriteAll is called, unless SetOutOnWrite 
		///  is set to <c>false</c>.</remarks>
		public TextWriter Out
		{
			get{ return _out; }
			set
			{ 
				_out = value; 
				foreach(ExecutionTimer timer in _timers)
				{
					timer.Out = _out;
				}
			}
		}

		/// <summary>
		/// Gets or sets the value indicating whether or not the Out property of
		///  each <see cref="ExecutionTimer"/> object being managed to the value 
		/// of this instance's Out property when WriteAll is called.
		/// </summary>
		public bool SetOutOnWrite
		{
			get{ return _setOutOnWrite; }
			set{ _setOutOnWrite = value; }
		}

		/// <summary>
		/// Gets or sets the BeginPrintString property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public PrintString BeginPrintString
		{
			get{ return _beginPrintString; }
			set
			{ 
				_beginPrintString = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.BeginPrintString = _beginPrintString;
				}
			}
		}

		/// <summary>
		/// Gets or sets the EndPrintString property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public PrintString EndPrintString
		{
			get{ return _endPrintString; }
			set
			{
				_endPrintString = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.EndPrintString = _endPrintString;
				}
			}
		}

		/// <summary>
		/// Gets or sets the SeriesBeginPrintString property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public PrintString SeriesBeginPrintString
		{
			get{ return _seriesBeginPrintString; }
			set
			{
				_seriesBeginPrintString = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.SeriesBeginPrintString = _seriesBeginPrintString;
				}
			}
		}

		/// <summary>
		/// Gets or sets the SeriesEndPrintString property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public PrintString SeriesEndPrintString
		{
			get{ return _seriesEndPrintString; }
			set
			{ 
				_seriesEndPrintString = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.SeriesEndPrintString = _seriesEndPrintString;
				}
			}
		}

		/// <summary>
		/// Gets or sets the Delegate object to be invoked when the execution 
		/// results for the <see cref="ExecutionTimer"/> objects are about 
		/// to be written.
		/// </summary>
		public PrintString CollectionBeginPrintString
		{
			get{ return _collectionBeginPrintString; }
			set{ _collectionBeginPrintString = value; }
		}

		/// <summary>
		/// Gets or sets the Delegate object to be invoked when the execution 
		/// results for the <see cref="ExecutionTimer"/> objects have been written.
		/// </summary>
		public PrintString CollectionEndPrintString
		{
			get{ return _collectionEndPrintString; }
			set{ _collectionEndPrintString = value; }
		}

		/// <summary>
		/// Gets or sets the BeginDelim property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public string BeginDelim
		{
			get{ return _beginDelim; }
			set
			{ 
				_beginDelim = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.BeginDelim = _beginDelim;
				}
			}
		}

		/// <summary>
		/// Gets or sets the EndDelim property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public string EndDelim
		{
			get{ return _endDelim; }
			set
			{ 
				_endDelim = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.EndDelim = _endDelim;
				}
			}
		}

		/// <summary>
		/// Gets or sets the SeriesBeginDelim property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public string SeriesBeginDelim
		{
			get{ return _seriesBeginDelim; }
			set
			{
				_seriesBeginDelim = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.SeriesBeginDelim = _seriesBeginDelim;
				}
			}
		}

		/// <summary>
		/// Gets or sets the SeriesEndDelim property of each <see cref="ExecutionTimer"/> 
		/// being managed.
		/// </summary>
		public string SeriesEndDelim
		{
			get{ return _seriesEndDelim; }
			set
			{ 
				_seriesEndDelim = value; 

				foreach(ExecutionTimer timer in _timers)
				{
					timer.SeriesEndDelim = _seriesEndDelim;
				}
			}
		}

		/// <summary>
		/// Gets or set the string to be written before the execution results
		///  for the <see cref="ExecutionTimer"/> objects being managed are written.
		/// </summary>
		public string CollectionBeginDelim
		{
			get{ return _collectionBeginDelim; }
			set{ _collectionBeginDelim = value; }
		}

		/// <summary>
		/// Gets or set the string to be written after the execution results
		///  for the <see cref="ExecutionTimer"/> objects being managed are written.
		/// </summary>
		public string CollectionEndDelim
		{
			get{ return _collectionEndDelim; }
			set{ _collectionEndDelim = value; }
		}

		/// <summary>
		/// Calls WriteSeries for each <see cref="ExecutionTimer"/> object being managed.
		/// </summary>
		public void WriteAll()
		{
			// It may seem strange to explicitly set a property with its own backing field, but
			// the setter for this particular property iterates over the timers in the
			// _timers collection and sets all of them to the value passed to the property.
			if(SetOutOnWrite)
			{
				Out = _out;
			}

			_out.WriteLine(GetCollectionBeginDelim());

			foreach(ExecutionTimer timer in _timers)
			{
				timer.WriteSeries();
			}

			_out.WriteLine(GetCollectionEndDelim());
		}

		// Returns the value returned by the method targeted by the 
		// CollectionBeginPrintString property.  Returns the string contained
		// in the CollectionBeginDelim property if ConnectionBeginPrintString is null.
		private string GetCollectionBeginDelim()
		{
			if(_collectionBeginPrintString != null)
			{
				return (string)_collectionBeginPrintString.DynamicInvoke(new object[]{});
			}
			else
			{
				return _collectionBeginDelim;
			}
		}

		// Returns the value returned by the method targeted by the 
		// CollectionEndPrintString property.  Returns the string contained
		// in the CollectionEndDelim property if ConnectionEndPrintString is null.
		private string GetCollectionEndDelim()
		{
			if(_collectionEndPrintString != null)
			{
				return (string)_collectionEndPrintString.DynamicInvoke(new object[]{});
			}
			else
			{
				return _collectionEndDelim;
			}
		}

		
		/// <summary>
		/// Calls RunTimedMethod on each <see cref="ExecutionTimer"/> 
		/// currently being managed.
		/// </summary>
		/// <param name="continueOnException">If <c>true</c>, exceptions 
		/// thrown will not cause execution of this method to stop.</param>
		public void ExecuteTimers(bool continueOnException)
		{
			// All other overloads of ExecuteTimer (and
			// AddAndExecuteTimer static methods) will end up here.

			for(int i = 0; i < _timers.Count; i++)
			{
				try
				{
					_timers[i].RunTimedMethod();
				}
				catch(SeriesAbortedException)
				{
					if(!continueOnException)
						throw;
				}
			}
		}

		/// <summary>
		/// Adds <see cref="ExecutionTimer"/> objects with the specified
		/// <see cref="TimedMethod"/> objects and calls RunTimedMethod
		/// on each of them.
		/// </summary>
		/// <param name="timedMethods">Adds <see cref="ExecutionTimer"/> objects with the specified
		/// <see cref="TimedMethod"/> objects and calls RunTimedMethod
		/// on each of them.</param>
		public void ExecuteTimers(TimedMethodCollection timedMethods)
		{
			ExecuteTimers(timedMethods, false);
		}

		/// <summary>
		/// Adds <see cref="ExecutionTimer"/> objects with the specified
		/// <see cref="TimedMethod"/> objects and calls RunTimedMethod
		/// on each of them.
		/// </summary>
		/// <param name="timedMethods">Adds <see cref="ExecutionTimer"/> objects with the specified
		/// <see cref="TimedMethod"/> objects and calls RunTimedMethod
		/// on each of them.</param>
		/// <param name="continueOnException">If <c>true</c>, exceptions 
		/// thrown will not cause execution of this method to stop.</param>
		public void ExecuteTimers(TimedMethodCollection timedMethods, bool continueOnException)
		{
			// Used to create new timers to add to the manager
			ExecutionTimer timer;
			string timerName = string.Empty;
			bool goodName = true;

			for(int i = 0; i < timedMethods.Count; i++)
			{
				timerName = timedMethods[i].DisplayName;
				
				if(timerName == null ||
					timerName.Length == 0)
				{
					// The purpose of this is to get a unique name, but not one that
					// is complete jibberish. I orginally had an algorithm that 
					// would first get the name of the method targeted by the 
					// TimedMethod.Method delegate and append a number (surrounded 
					// by parens) to it.  I changed it to this slower algorithm 
					// for a couple of reasons: 1) The first one required me to 
					// check the Method property of to make sure it points to a 
					// valid method before I retrieved the name.  It's possible 
					// for it to be null, which obviously wouldn't be kosher, but 
					// I didn't want to move the responsibility of validating the
					// contents of the TimedMethod into this method, because that's
					// already taken care of in the ExecutionTimer.RunTimedMethod
					// method.  2) There is a chance that the previously mentioned
					// naming scheme would pick a name that would appear later in the 
					// TimedMethodCollection, in which case that name would then be 
					// changed, which could then require a name change further on
					// down the line.  I suppose it is still statistically possible 
					// for that to happen, but the chances are so low that I'm not going to 
					// worry about it.
					// Obviously, this won't run if all the TimedMethods have DisplayNames.
					//
					// Did I really just write a 19 line comment? <-- Warning: This line is not self-aware
					do
					{
						goodName = true;

						timerName = "Timer ID: " + new Random(DateTime.Now.Millisecond).Next().ToString();

						for(int j = 0; j < timedMethods.Count; j++)
						{
							if(timedMethods[j].DisplayName == timerName)
								goodName = false;				
						}
					}while(!goodName);
				}

				timer = new ExecutionTimer(timerName);

				// Assign main, setup, and teardown delegates and their arguments
				timer.TimedMethod = timedMethods[i];
	
				try
				{
					if(NameTaken(timerName))
						throw new NameExistsException("This manager is already managing a timer by this name", timerName);
				}
				catch(NameExistsException)
				{
					if(!continueOnException)
						throw;
				}
				AddTimer(timer);				
			}

			ExecuteTimers(continueOnException);
		}

		/// <summary>
		/// Adds the <see cref="TimedMethod"/> objects in the specified 
		/// <see cref="TimedMethodCollection"/> object to an <see cref="ExecutionTimerManager"/> 
		/// and returns the resulting <see cref="ExecutionTimerManager"/> after
		///  executing the methods.
		/// </summary>
		/// <param name="timedMethods">The <see cref="TimedMethod"/> objects 
		/// to be managed.</param>
		/// <returns>An <see cref="ExecutionTimerManager"/> object which 
		/// manages a collection of <see cref="ExecutionTimer"/> objects whose 
		/// <see cref="TimedMethod"/> objects have already beenm run.</returns>
		/// <exception cref="NameExistsException">A timer with the name specified by 
		/// the DisplayName property of a <see cref="TimedMethod"/> object 
		/// already exists in the collection.</exception>
		/// <exception cref="SeriesAbortedException">The method being timed, or its
		///  set-up or tear-down method threw an exception.</exception>
		public static ExecutionTimerManager AddAndExecuteTimers(TimedMethodCollection timedMethods)
		{
			return AddAndExecuteTimers(timedMethods, false);
		}

		/// <summary>
		/// Adds the <see cref="TimedMethod"/> objects in the specified 
		/// <see cref="TimedMethodCollection"/> object to an <see cref="ExecutionTimerManager"/> 
		/// and returns the resulting <see cref="ExecutionTimerManager"/> after executing the 
		/// methods.
		/// </summary>
		/// <param name="timedMethods">The <see cref="TimedMethod"/> objects 
		/// to be managed.</param>
		/// <param name="continueOnException">If true, any exceptions thrown by 
		/// the <see cref="ExecutionTimer"/> objects will cause the offending
		///  timer to be skipped, rather than causing this method to fail.</param>
		/// <returns>An <see cref="ExecutionTimerManager"/> object which 
		/// manages a collection of <see cref="ExecutionTimer"/> objects whose 
		/// <see cref="TimedMethod"/> objects have already beenm run.</returns>
		/// <exception cref="NameExistsException">A timer with the name specified by 
		/// the DisplayName property of a <see cref="TimedMethod"/> object 
		/// already exists in the collection.  This exception will be swallowed if
		///   <c>true</c> is passed as the second argument to this method.</exception>
		/// <exception cref="SeriesAbortedException">The method being timed, or its
		///  set-up or tear-down method threw an exception.  This exception will be swallowed if
		///   <c>true</c> is passed as the second argument to this method.</exception>
		public static ExecutionTimerManager AddAndExecuteTimers(TimedMethodCollection timedMethods, bool continueOnException)
		{
			ExecutionTimerManager manager = new ExecutionTimerManager();

			manager.ExecuteTimers(timedMethods, continueOnException);

			return manager;
		}
	}
}