using System;
using System.IO;
using System.Text;

namespace HarryBonesSolutions.Performance
{
	/// <summary>
	/// This class is used to measure the execution time of arbitrary methods.
	/// </summary>
	/// <remarks>This class can be used in several ways.  The simplest use is
	/// to call the <c>static</c> <see cref="TimeBlock"/> method in a 
	/// C# <c>using statement.</c>  (This can be accomplished in other 
	/// languanges by calling the <see cref="Dispose"/> method manually.)
	/// <br/>
	/// The second way to use this class is to instantiate it and call
	/// its methods directly to measure the execution time for a given
	///  method.  Every time the same timer is started and stopped, statistics 
	/// for the series are updated.  These statistics can be cleared by calling 
	/// <see cref="ClearRunHistory"/>.
	/// <br/>
	/// The third way is to specify a <see cref="TimedMethod"/> object 
	/// which contains a method to measure, number of times to run it,
	/// and (optionally) set-up and tear-down methods.
	/// <br/>
	/// With the second and third methods, relatively fine-grained control
	///  is given to the caller for how and where the output is written.  This
	/// control is exercised through the <see cref="BeginDelim"/>, 
	/// <see cref="EndDelim"/>, <see cref="SeriesBeginDelim"/>, and 
	/// <see cref="SeriesEndDelim"/> string properties, the <see cref="BeginPrintString"/>, 
	/// <see cref="EndPrintString"/>, <see cref="SeriesBeginPrintString"/>, 
	/// <see cref="SeriesEndPrintString"/> <see cref="PrintString"/> delegate 
	/// properties, and the <see cref="ResultsPrinting"/>, <see cref="ResultsPrinted"/>, 
	/// <see cref="SeriesResultsPrinting"/>, and <see cref="SeriesResultsPrinted"/> 
	/// events.</remarks>
	/// <example>
	/// The following example shows all three uses.
	/// <code>
	/// public class Example
	/// {
	/// 	public static void Main()
	/// 	{
	/// 		Example example = new Example();
	/// 		example.TimeMethods();
	/// 	}
	/// 	
	/// 	// This is the method we'll be timing
	/// 	private void TakeOneSecond()
	/// 	{
	/// 		System.Threading.Thread.Sleep(1000);
	/// 	}
	/// 
	/// 	private void TimeMethods()
	/// 	{
	/// 		// Time it using the using statement with the TimeBlock method.
	/// 		using(ExecutionTimer.TimeBlock("Method 1"))
	/// 		{
	/// 			TakeOneSecond();
	/// 		}
	/// 
	/// 		// Time it by instantiating an ExecutionTimer object and 
	/// 		// calling its methods.
	/// 		ExecutionTimer timer = new ExecutionTimer("Method 2");
	/// 
	/// 		// Set the string to be printed before the results are printed.
	/// 		timer.BeginPrintString = "------****------";
	/// 
	/// 		timer.Start();
	/// 
	/// 		// Run the method being timed.
	/// 		TakeOneSecond();
	/// 
	/// 		timer.Stop();
	/// 
	/// 		// If timer.AutoTrace were set to true, this would
	/// 		// have been called automatically when the timer was stopped.
	/// 		timer.Write();
	/// 
	/// 		// Add a few more runs to the same timer
	/// 		for(int i = 0; i &lt; 3; i++)
	/// 		{
	/// 			timer.Start();
	/// 	
	/// 			TakeOneSecond();
	/// 	
	/// 			timer.Stop();
	/// 		}
	/// 
	/// 		// Now write the results for the series
	/// 		timer.WriteSeries();
	/// 
	/// 
	/// 		// Time it by passing delegates targeting the method to time
	/// 		// as well as customize the output a bit
	/// 		TimeMethodsWithDelegates();
	/// 	}
	/// 
	/// 	private string GetOutputHeader(string timerName)
	/// 	{
	/// 		string output = "Executing " + timerName + " on ";
	/// 	
	/// 		output += DateTime.Now.ToLongDateString();
	/// 	
	/// 		output += " at ";
	/// 	
	/// 		outout += DateTime.Now.ToLongTimeString();
	/// 	
	/// 		return output;
	/// 	}
	/// 
	/// 	private void BeforeOutputWritten(object sender, ExecutionTimerEventArgs e)
	/// 	{
	/// 		// I'd like to handle the output for different timers differently.
	/// 	
	/// 		ExecutionTimer timer = (ExecutionTimer)sender;
	/// 	
	/// 		if(e.TimerName.StartsWith("Test"))
	/// 			timer.Out = Console.Out;
	/// 		else
	/// 			timer.Out = writer; // writer is a TextWriter defined elsewhere
	/// 	}
	/// 
	/// 	private void AfterOutputWritten(object sender, ExecutionTimerEventArgs e)
	/// 	{
	/// 		// We need to clean up if our TextWriter was used
	///   
	/// 		if(e.TimerName.StartsWith("Test") &amp;&amp;
	/// 			writer != null)
	/// 			writer.Close();
	/// 	}
	/// 
	/// 	private void TimeMethodsWithDelegates()
	/// 	{
	/// 		// This method of using ExecutionTimer is geared towards automated 
	/// 		// suites and the like where you can set up methods to run from 
	/// 		// configuration files or other sources.  It is usually used
	/// 		// in conjuction with an ExecutionTimerManager object instead of
	/// 		// directly, but it can be used alone as shown here.
	/// 	
	/// 		ExecutionTimer timer = new ExecutionTimer("Method 3");
	/// 	
	/// 		// Print out the results of each run automatically
	/// 		timer.AutoTrace = true;
	/// 	
	/// 		timer.BeginPrintString = new PrintString(GetOutputHeader);
	/// 	
	/// 		// Hook up our event handlers
	/// 		timer.ResultsPrinting += new ExecutionTimerEventHandler(BeforeOutputWritten);
	/// 	
	/// 		timer.ResultsPrinted += new ExecutionTimerEventHandler(AfterOutputWritten);
	/// 	
	/// 		// Specify the delegates to be run
	/// 	
	/// 		TimedMethod timedMethod = new TimedMethod();
	/// 	
	/// 		timedMethod.Method = new MethodInvoker(TakeOneSecond);
	/// 	
	/// 		timedMethod.Iterations = 3;
	/// 	
	/// 		// If the timed method needs arguments, we can pass them
	/// 		// here.  Also, if the timed method changes the state of an 
	/// 		// object, we can specify set-up and tear-down methods to restore
	/// 		// the state for each run.  Those methods can also have arguments
	/// 		/// specified.
	/// 	
	/// 		timer.TimedMethod = timedMethod;
	/// 	
	/// 		timer.RunTimedMethod();
	/// 	}
	/// } 
	/// 
	/// Here is the output produced by the preceding code:
	/// 
	/// 	***********************
	/// 	Execution time for "Method 1" (Run 1): 1000 ms.
	/// 
	/// 	------****------
	/// 	Execution time for "Method 2" (Run 1): 1000 ms.
	/// 
	/// 	***********************
	/// 	Stats for "Method 2":
	/// 	- Number of runs completed: 4
	/// 	- Average execution time per run: 1000 ms.
	/// 	- Shortest run: 1000 ms.
	/// 	- Longest run: 1000 ms.
	/// 
	/// 	Executing Method 3 on Friday, October 22, 2004 at 4:18:35 PM
	/// 	Execution time for "Method 3" (Run 1): 1000 ms.
	/// 
	/// 	Executing Method 3 on Friday, October 22, 2004 at 4:18:36 PM
	/// 	Execution time for "Method 3" (Run 2): 1000 ms.
	/// 
	/// 	Executing Method 3 on Friday, October 22, 2004 at 4:18:37 PM
	/// 	Execution time for "Method 3" (Run 3): 1000 ms.
	/// 
	/// 	***********************
	/// 	Stats for "Method 3":
	/// 	- Number of runs completed: 3
	/// 	- Average execution time per run: 1000 ms.
	/// 	- Shortest run: 1000 ms.
	/// 	- Longest run: 1000 ms.
	/// </code>	
	/// </example>
	public class ExecutionTimer : IDisposable
	{
		#region Private Fields

		private string _beginDelim;
		private string _endDelim;
		private string _seriesBeginDelim;
		private string _seriesEndDelim;
		private string _name;
		private long	_start;
		private long	_end;
		private long	_totalExecutionTime;
		private int		_numberOfRuns;
		private bool	_autoTrace;
		private bool	_running;
		private TextWriter  _out;
		private TimeSpan	  _longestRun;
		private TimeSpan	  _shortestRun;
		private PrintString _beginPrintString;
		private PrintString _endPrintString;
		private PrintString _seriesBeginPrintString;
		private PrintString _seriesEndPrintString;
		private TimedMethod _timedMethod;

		#endregion

		#region Constructors

		/// <summary>
		/// Initializes a new instance of the ExecutionTimer class.
		/// </summary>
		public ExecutionTimer()
		{
			_name	= string.Empty;
			_endDelim = string.Empty;
			_seriesEndDelim = string.Empty;
			_end = 0;
			_start = 0;
			_numberOfRuns = 0;
			_totalExecutionTime = 0;
			_running	= false;
			_autoTrace = false;
			_timedMethod = null;
			_endPrintString = null;
			_beginPrintString = null;
			_seriesEndPrintString = null;
			_seriesBeginPrintString = null;
			_out = Console.Out;
			_longestRun	= TimeSpan.Zero;
			_shortestRun = TimeSpan.Zero;
			_beginDelim = "***********************";
			_seriesBeginDelim = "***********************";			
		}

		/// <summary>
		/// Initializes a new instance of the ExecutionTimer class.
		/// </summary>
		/// <param name="name">Name (or description) of the timer instance.</param>
		public ExecutionTimer(string name) : this()
		{
			_name = name;
		}

		/// <summary>
		/// Initializes a new instance of the ExecutionTimer class.
		/// </summary>
		/// <param name="name">Name (or description) of the timer instance.</param>
		/// <param name="autoTrace">If true, Write and WriteSeries will be called automatically when Stop and
		/// RunTimedMethod, respectively, are called.</param>
		public ExecutionTimer(string name, bool autoTrace) : this(name)
		{
			_autoTrace = autoTrace;
		}

		/// <summary>
		/// Initializes a new instance of the ExecutionTimer class.
		/// </summary>
		/// <param name="name">Name (or description) of the timer instance.</param>
		/// <param name="timedMethod">The delegate targeting the method to execute when
		/// RunTimedMethod is called.</param>
		public ExecutionTimer(string name, TimedMethod timedMethod) : this(name)
		{
			_timedMethod = TimedMethod;
		}

		/// <summary>
		/// Initializes a new instance of the ExecutionTimer class.
		/// </summary>
		/// <param name="name">Name (or description) of the timer instance.</param>
		/// <param name="timedMethod">The delegate targeting the method to execute when
		/// RunTimedMethod is called.</param>
		/// <param name="autoTrace">If true, Write and WriteSeries will be called automatically when Stop and
		/// RunTimedMethod, respectively, are called.</param>
		public ExecutionTimer(string name, TimedMethod timedMethod, bool autoTrace) : this(name, timedMethod)
		{
			_autoTrace = autoTrace;
		}

		#endregion

		#region Properties

		/// <summary>
		/// Gets or sets the the TextWriter the output will be written to.  Default is set to System.Console.Out
		/// </summary>
		public TextWriter Out
		{
			get{ return _out; }
			set{ _out = value; }
		}

		/// <summary>
		/// Gets or sets the string to be printed before the output for each run.
		/// </summary>
		public string BeginDelim
		{
			get{ return _beginDelim; }
			set{ _beginDelim = value; }
		}

		/// <summary>
		/// Gets or sets the string to be printed after the output for each run.
		/// </summary>
		public string EndDelim
		{
			get{ return _endDelim; }
			set{ _endDelim = value; }
		}

		/// <summary>
		/// Gets or sets the string to be printed before series stats are output.
		/// </summary>
		public string SeriesBeginDelim
		{
			get{ return _seriesBeginDelim; }
			set{ _seriesBeginDelim = value; }
		}

		/// <summary>
		/// Gets or sets the string to be printed after series stats are output.
		/// </summary>
		public string SeriesEndDelim
		{
			get{ return _seriesEndDelim; }
			set{ _seriesEndDelim = value; }
		}

		/// <summary>
		/// Gets or sets the PrintString delegate to be executed before the output for each run is written.  
		/// 
		/// If not null, the return value from this delegate is used in place of the string 
		/// contained in the BeginDelim property.  If this delegate targets more then one method, only the 
		/// first will be called.
		/// </summary>
		public PrintString BeginPrintString
		{
			get{ return _beginPrintString; }
			set{ _beginPrintString = value; }
		}

		/// <summary>
		/// Gets or sets the PrintString delegate to be executed after the output for each run is written.  
		/// 
		/// If not null, the return value from this delegate is used in place of the string 
		/// contained in the EndDelim property.  If this delegate targets more then one method, only the 
		/// first will be called.
		/// </summary>
		public PrintString EndPrintString
		{
			get{ return _endPrintString; }
			set{ _endPrintString = value; }
		}

		/// <summary>
		/// Gets or sets the PrintString delegate to be executed before the series stats are written.
		/// 
		/// If not null, the return value from this delegate is used in place of the string
		/// contained in the SeriesBeginDelim property.  If this delegate targets more then one method, 
		/// only the first will be called.
		/// </summary>
		public PrintString SeriesBeginPrintString
		{
			get{ return _seriesBeginPrintString; }
			set{ _seriesBeginPrintString = value; }
		}

		/// <summary>
		/// Gets or sets the PrintString delegate to be executed after the series stats are written.
		/// 
		/// If not null, the return value from this delegate is used in place of the string
		/// contained in the SeriesEndDelim property.  If this delegate targets more then one method, 
		/// only the first will be called.
		/// </summary>
		public PrintString SeriesEndPrintString
		{
			get{ return _seriesEndPrintString; }
			set{ _seriesEndPrintString = value; }
		}

		/// <summary>
		/// Gets the name for (or description) of this instance.  This is included in any output.
		/// </summary>
		public string Name
		{
			get{ return _name; }
		}

		/// <summary>
		/// Gets the TimeSpan representing execution time of the longest run in the series.
		/// </summary>
		public TimeSpan LongestRun
		{
			get{ return _longestRun; }
		}

		/// <summary>
		/// Gets the TimeSpan representing execution time of the shortest run in the series.
		/// </summary>																			  
		public TimeSpan ShortestRun
		{
			get{ return _shortestRun; }
		}

		/// <summary>
		/// Gets or sets the value indicating whether to automatically call the Write method when 
		/// the timer is stopped and WriteSeries after all iterations have been completed in 
		/// RunTimedMethod.
		/// </summary>
		public bool AutoTrace
		{
			get{ return _autoTrace; }
			set{ _autoTrace = value; }
		}

		/// <summary>
		/// Gets the number of times this instance has been started and stopped.
		/// </summary>
		public int RunCount
		{
			get{ return _numberOfRuns; }
		}

		/// <summary>
		/// Gets or sets the <see cref="TimedMethod"/> object whcih specifies 
		/// the method to be timed.
		/// </summary>
		public TimedMethod TimedMethod
		{
			get{ return _timedMethod; }
			set{ _timedMethod = value; }
		}

		#endregion

		#region Events

		/// <summary>
		/// Occurs when the timer is about to be started.
		/// </summary>
		public event ExecutionTimerEventHandler TimerStarting;

		/// <summary>
		/// Occurs when the timer has been recorded.
		/// </summary>
		/// <remarks>This event is fired after the timer's stop time
		/// has been recorded and all statistics have been updated, but before
		/// any output has been written (if AutoTrace is set to true).</remarks>
		public event ExecutionTimerStoppedEventHandler TimerStopped;

		/// <summary>
		/// Occurs when results for a run are about to be printed.
		/// </summary>
		/// <remarks>This event is fired before any output is
		/// written for the run.</remarks>
		public event ExecutionTimerEventHandler ResultsPrinting;

		/// <summary>
		/// Occurs after results for a run are printed.
		/// </summary>
		/// <remarks>This event is fired after all output is
		/// written for the run.</remarks>
		public event ExecutionTimerEventHandler ResultsPrinted;

		/// <summary>
		/// Occurs when results for a series are about to be printed.
		/// </summary>
		/// <remarks>This event is fired before any output is
		/// written for the series.</remarks>
		public event ExecutionTimerEventHandler SeriesResultsPrinting;

		/// <summary>
		/// Occurs after results for a series are printed.
		/// </summary>
		/// <remarks>This event is fired after all output is
		/// written for the series.</remarks>
		public event ExecutionTimerEventHandler SeriesResultsPrinted;

		#endregion

		#region Public Instance Methods

		/// <summary>
		/// Resets RunCount, LongestRun, ShortestRun to their initial values.
		/// </summary>
		public void ClearRunHistory()
		{
			_running = false;
			_start = 0;
			_end = 0;
			_numberOfRuns = 0;
			_totalExecutionTime = 0;
			_longestRun = TimeSpan.Zero;
			_shortestRun = TimeSpan.Zero;
		}

		/// <summary>
		/// Runs the method targeted by the TimedMethod delegate with no arguments for the
		/// specified number of iterations.
		/// </summary>
		/// <exception cref="SeriesAbortedException">One of the methods specified
		///  by the set-up timed method, or tear-down
		///  delegates failed.  This could be caused by the method not being found,
		///  the target method throwing an exception, the caller not haing the
		///  required permissions to call the method, or the incorrect aruments being 
		///   passed to the delegate method.</exception>
		public void RunTimedMethod()
		{
			RunTimedMethod(_timedMethod.Iterations);
		}

		/// <summary>
		/// Runs the methods targeted by the delegates specified in the
		/// TimedMethod object with the specified arguments.
		/// </summary>
		/// <param name="timedMethod">TimedMethod containing the delegates
		/// targeting the methods to be executed, along with any arguments to
		/// pe passed to those methods.  The TimedMethod property will
		/// be set to the value of this argument.</param>
		/// <exception cref="SeriesAbortedException">One of the methods specified
		///  by the set-up timed method, or tear-down
		///  delegates failed.  This could be caused by the method not being found,
		///  the target method throwing an exception, the caller not haing the
		///  required permissions to call the method, or the incorrect aruments being 
		///   passed to the delegate method.</exception>
		public void RunTimedMethod(TimedMethod timedMethod)
		{
			if(_timedMethod != timedMethod)
			{
				_timedMethod = timedMethod;
			}

			RunTimedMethod(_timedMethod.Iterations);
		}

		/// <summary>
		/// Runs the methods targeted by the delegates specified in the
		/// TimedMethod object with the specified arguments for the 
		/// specified number of iterations.
		/// </summary>
		/// <param name="timedMethod">TimedMethod containing the delegates
		/// targeting the methods to be executed, along with any arguments to
		/// pe passed to those methods.  The TimedMethod property will
		/// be set to the value of this argument.</param>
		/// <param name="iterations">The number of times to run the timed method.  
		/// This will not set the Iterations property of the TimedMethod object.</param>
		/// <exception cref="SeriesAbortedException">One of the methods specified
		///  by the set-up timed method, or tear-down
		///  delegates failed.  This could be caused by the method not being found,
		///  the target method throwing an exception, the caller not haing the
		///  required permissions to call the method, or the incorrect aruments being 
		///   passed to the delegate method.</exception>
		public void RunTimedMethod(TimedMethod timedMethod, int iterations)
		{

			if(_timedMethod != timedMethod)
			{
				_timedMethod = timedMethod;
			}

			RunTimedMethod(iterations);
		}

		/// <summary>
		/// Runs the method targeted by the TimedMethod delegate with no arguments for the
		/// specified number of iterations.
		/// </summary>
		/// <param name="iterations">The number of times to run the timed method.  This will not
		/// set the Iterations property of the TimedMethod object.</param>
		/// <exception cref="SeriesAbortedException">One of the methods specified
		///  by the set-up timed method, or tear-down
		///  delegates failed.  This could be caused by the method not being found,
		///  the target method throwing an exception, the caller not haing the
		///  required permissions to call the method, or the incorrect aruments being 
		///   passed to the delegate method.</exception>
		public void RunTimedMethod(int iterations)
		{
			// Note: The actual implementation of this method is in an overload of RunTimedMethod
			// that accepts an int parameter so that the number of iterations to perform
			// are settable at the method level without having to alter the Iterations property
			// of the TimedMethod being run.  I thought it may be helpful to do so in certain
			// situations.

			// These exist because I don't want to alter the input parameters
			object[] localArgs;
			object[] localSetUpArgs;
			object[] localTearDownArgs;

			// validate arguments
			if(_timedMethod == null)
				throw new InvalidOperationException("_timedMethod cannot be null");

			//If any of the args are null, create an empty object array.
			localArgs = _timedMethod.Args == null ? new object[]{} : _timedMethod.Args;

			localSetUpArgs = _timedMethod.SetUpArgs == null ? new object[]{} : _timedMethod.SetUpArgs;

			localTearDownArgs = _timedMethod.TearDownArgs == null ? new object[]{} : _timedMethod.TearDownArgs;

			// Need at least one iteration to execute.
			if(iterations < 1)
				throw new ArgumentOutOfRangeException("iterations", iterations, "Value must be greater than or equal to 1");

			for(int i = 0; i < iterations; i++)
			{
				if(_timedMethod.SetUp != null)
				{
					// A set-up method has been specified.  Try to run it.
					try
					{
						_timedMethod.SetUp.GetInvocationList()[0].DynamicInvoke(localSetUpArgs);
					}
					catch(ArgumentException ex)
					{
						ClearRunHistory();
						throw new SeriesAbortedException("Series aborted at TimedMethod.SetUp.  Run history cleared.", _name, ex);
					}
					catch(MemberAccessException ex)
					{
						ClearRunHistory();
						throw new SeriesAbortedException("Series aborted at TimedMethod.SetUp.  Run history cleared.", _name, ex);
					}
					catch(System.Reflection.TargetException ex)
					{
						ClearRunHistory();
						throw new SeriesAbortedException("Series aborted at TimedMethod.SetUp.  Run history cleared.", _name, ex);
					}
					catch(System.Reflection.TargetInvocationException ex)
					{
						ClearRunHistory();
						throw new SeriesAbortedException("Series aborted at TimedMethod.SetUp.  Run history cleared.", _name, ex);
					}
				}

				// The timer is only run after the set-up method has completed.
				Start();

				try
				{
					// Run the main method to be timed.
					_timedMethod.Method.GetInvocationList()[0].DynamicInvoke(localArgs);

					// Stop the timer before running the tear-down method.
					Stop();

					if(_timedMethod.TearDown != null)
					{
						// A tear-down method has been specified.  Try to run it.
						try
						{
							_timedMethod.TearDown.GetInvocationList()[0].DynamicInvoke(localTearDownArgs);
						}
						catch(ArgumentException ex)
						{
							ClearRunHistory();
							throw new SeriesAbortedException("Series aborted at TimedMethod.TearDown.  Run history cleared.", _name, ex);
						}
						catch(MemberAccessException ex)
						{
							ClearRunHistory();
							throw new SeriesAbortedException("Series aborted at TimedMethod.TearDown.  Run history cleared.", _name, ex);
						}
						catch(System.Reflection.TargetException ex)
						{
							ClearRunHistory();
							throw new SeriesAbortedException("Series aborted at TimedMethod.TearDown.  Run history cleared.", _name, ex);
						}
						catch(System.Reflection.TargetInvocationException ex)
						{
							ClearRunHistory();
							throw new SeriesAbortedException("Series aborted at TimedMethod.TearDown.  Run history cleared.", _name, ex);
						}
					}
				}
				catch(ArgumentException ex)
				{
					ClearRunHistory();
					throw new SeriesAbortedException("Series aborted at TimedMethod.Method.  Run history cleared.", _name, ex);
				}
				catch(MemberAccessException ex)
				{
					ClearRunHistory();
					throw new SeriesAbortedException("Series aborted at TimedMethod.Method.  Run history cleared.", _name, ex);
				}
				catch(System.Reflection.TargetException ex)
				{
					ClearRunHistory();
					throw new SeriesAbortedException("Series aborted at TimedMethod.Method.  Run history cleared.", _name, ex);
				}
				catch(System.Reflection.TargetInvocationException ex)
				{
					ClearRunHistory();
					throw new SeriesAbortedException("Series aborted at TimedMethod.Method.  Run history cleared.", _name, ex);
				}					
			}

			if(_autoTrace)
				WriteSeries();
		}

		/// <summary>
		/// Returns the execution time of the last run.
		/// </summary>
		/// <returns>TimeSpan</returns>
		/// <exception cref="InvalidOperationException">The timer is
		///  currently running.</exception>
		public TimeSpan GetExecutionTime()
		{
			// _end - _start will return a negative value if the timer
			// is running, which doesn't make much sense.
			if(_running)
				throw new InvalidOperationException("Cannot get execution time "
					+ "while timer is running.");
					
			long ticks = _end - _start;

			return new TimeSpan(ticks);
		}

		/// <summary>
		/// Returns the average execution time for all runs.
		/// </summary>
		/// <returns>TimeSpan</returns>
		public TimeSpan GetSeriesAverageExecutionTime()
		{
			// Just return a zero length TimeSpan if no runs have been completed.
			// Also guards against potential divide-by-zero exception on the next line.
			if(_numberOfRuns == 0)
				return TimeSpan.Zero;

			long averageRun = _totalExecutionTime / _numberOfRuns;

			return new TimeSpan(averageRun);
		}

		/// <summary>
		/// Starts the timer.
		/// </summary>
		public void Start()
		{
			// Take no action if the timer is already running
			if(_running) return;

			_running = true;

			OnTimerStarting(new ExecutionTimerEventArgs(Name));

			// Record start time just before returning
			_start = DateTime.Now.Ticks;
		}

		/// <summary>
		/// Stops the timer.  If AutoTrace is set to true, calls Write after timer is stopped.
		/// </summary>
		public void Stop()
		{
			// I'm using a temporary variable because I want to recod
			// the end time right away so the execution time isn't
			// being extended by this class' execution and I don't want to 
			// stomp any preexisting value if the timer is not actually running
			long tmpEnd = DateTime.Now.Ticks;

			// Take no action if Stop is called on a stopped timer
			if(!_running) return;

			_running = false;

			_end = tmpEnd;

			_numberOfRuns++;

			TimeSpan thisRun = GetExecutionTime();
			
			_totalExecutionTime += thisRun.Ticks;

			if(_numberOfRuns == 1)
			{
				// The first run will always be the longest and shortest.
				_shortestRun = thisRun;
				_longestRun = thisRun;
			}
			else
			{
				// See if this run was shorter or longer than any other.
				_shortestRun = thisRun < _shortestRun ? thisRun : _shortestRun;
				_longestRun = thisRun > _longestRun ? thisRun : _longestRun;
			}

			// Raise the TimerStopped event.
			OnTimerStopped(new ExecutionTimerDetailedEventArgs(Name, thisRun,
				_numberOfRuns, new TimeSpan(_totalExecutionTime), _shortestRun, 
				_longestRun));

			if(_autoTrace)
			{
				Write();
			}
		}

		/// <summary>
		/// Writes the timer's Name and total execution time (in milliseconds) to the 
		/// TextWriter specified in the Out property.
		/// </summary>
		public void Write()
		{
			if(_running)
				throw new InvalidOperationException("Write called on a running timer");

			// Raise the OnResultsPrinting event.
			OnResultsPrinting(new ExecutionTimerEventArgs(Name));

			string newLine = Environment.NewLine;
			
			// Get the execution time for the most recent run.
			TimeSpan execTime = GetExecutionTime();

			StringBuilder output = new StringBuilder();

			// Build the output.

			output.Append(GetBeginDelimString());
			output.Append(newLine);

			if(_numberOfRuns == 0)
			{
				output.Append("\"");
				output.Append(_name == string.Empty ? "<ExecutionTimer>" : _name);
				output.Append("\" has not yet completed a run.");
			}
			else
			{
				output.Append("Execution time for ");
				output.Append("\"");
				output.Append(_name == string.Empty ? "<ExecutionTimer>" : _name);
				output.Append("\" (Run ");
				output.Append(_numberOfRuns.ToString());
				output.Append("): ");
				output.Append(execTime.TotalMilliseconds.ToString());
				output.Append(" ms.");
			}

			output.Append(newLine);
			output.Append(GetEndDelimString());

			// Write the output to the TextWriter specified in the Out poperty.
			_out.WriteLine(output.ToString());

			// Raise the OnResultsPrinted event.
			OnResultsPrinted(new ExecutionTimerEventArgs(Name));
		}

		/// <summary>
		/// Writes the timer's Name, RunCount, LongestRun, ShortestRun, and average execution time
		/// to the TextWriter specified in the Out property.  All times are output in milliseconds.
		/// </summary>
		public void WriteSeries()
		{
			if(_running)
				throw new InvalidOperationException("WriteSeries called on a running timer");

			// Raise the SeriesResultsPrinting event.
			OnSeriesResultsPrinting(new ExecutionTimerEventArgs(Name));

			string newLine = Environment.NewLine;

			StringBuilder output = new StringBuilder();

			// Build the output.

			output.Append(GetSeriesBeginDelimString());
			output.Append(newLine);

			if(_numberOfRuns == 0)
			{
				output.Append(_name == string.Empty ? "<ExecutionTimer>" : _name);
				output.Append("\" has not yet completed a run.");
			}
			else
			{
				output.Append("Stats for \"");
				output.Append(_name == string.Empty ? "<ExecutionTimer>" : _name);
				output.Append("\":");
				output.Append(newLine);
				output.Append(" - Number of runs completed: ");
				output.Append(_numberOfRuns.ToString());
				output.Append(newLine);
				output.Append(" - Average execution time per run: ");
				output.Append(GetSeriesAverageExecutionTime().TotalMilliseconds.ToString());
				output.Append(" ms.");
				output.Append(newLine);
				output.Append(" - Shortest run: ");
				output.Append(_shortestRun.TotalMilliseconds.ToString());
				output.Append(" ms.");
				output.Append(newLine);
				output.Append(" - Longest run: ");
				output.Append(_longestRun.TotalMilliseconds.ToString());
				output.Append(" ms.");
			}

			output.Append(newLine);
			output.Append(GetSeriesEndDelimString());

			_out.WriteLine(output.ToString());

			// Raise the SeriesResultsPrinted event.
			OnSeriesResultsPrinted(new ExecutionTimerEventArgs(Name));
		}	

		/// <summary>
		/// Overrides object.ToString
		/// </summary>
		/// <returns>string</returns>
		public override string ToString()
		{
			return "ExecutionTimer: <" + _name + ">";
		}

		#endregion

		#region Public Static Methods

		/// <summary>
		/// This method exists to facilitate this class' use in the
		/// C# 'using' statement.
		/// </summary>
		/// <param name="description">Description of the timer to be printed.</param>
		/// <returns>IDisposable</returns>
		public static IDisposable TimeBlock(string description)
		{
			ExecutionTimer timer = new	ExecutionTimer(description);

			timer.AutoTrace = true;

			timer.Start();

			return timer as IDisposable;
		}

		#endregion

		#region Private Helper Methods

		// Returns a string retrieved either from the delegate contained 
		// in the BeginPrintString property or string contained in the
		// BeginDelim property if no delegate is specified.
		private string GetBeginDelimString()
		{
			if(_beginPrintString != null)
			{
				return (string)_beginPrintString.GetInvocationList()[0].DynamicInvoke(new object[]{Name});
			}
			else
			{
				return _beginDelim;
			}
		}

		// Returns a string retrieved either from the delegate contained 
		// in the SeriesBeginPrintString property or string contained in the
		// SeriesBeginDelim property if no delegate is specified.
		private string GetSeriesBeginDelimString()
		{
			if(_seriesBeginPrintString != null)
			{
				return (string)_seriesBeginPrintString.GetInvocationList()[0].DynamicInvoke(new object[]{Name});
			}
			else
			{
				return _seriesBeginDelim;
			}
		}

		// Returns a string retrieved either from the delegate contained 
		// in the EndPrintString property or string contained in the
		// EndDelim property if no delegate is specified.
		private string GetEndDelimString()
		{
			if(_endPrintString != null)
			{
				return (string)_endPrintString.GetInvocationList()[0].DynamicInvoke(new object[]{Name});
			}
			else
			{
				return _endDelim;
			}
		}

		// Returns a string retrieved either from the delegate contained 
		// in the SeriesEndPrintString property or string contained in the
		// SeriesEndDelim property if no delegate is specified.
		private string GetSeriesEndDelimString()
		{
			if(_seriesEndPrintString != null)
			{
				return (string)_seriesEndPrintString.GetInvocationList()[0].DynamicInvoke(new object[]{Name});
			}
			else
			{
				return _seriesEndDelim;
			}
		}

		#endregion

		#region Protected Methods

		/// <summary>
		/// Raises the TimerStarting event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnTimerStarting(ExecutionTimerEventArgs e)
		{
			ExecutionTimerEventHandler handler = TimerStarting;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		/// <summary>
		/// Raises the TimerStopped event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerDetailedEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnTimerStopped(ExecutionTimerDetailedEventArgs e)
		{
			ExecutionTimerStoppedEventHandler handler = TimerStopped;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		/// <summary>
		/// Raises the ResultsPrinting event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnResultsPrinting(ExecutionTimerEventArgs e)
		{
			ExecutionTimerEventHandler handler = ResultsPrinting;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		/// <summary>
		/// Raises the ResultsPrinted event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnResultsPrinted(ExecutionTimerEventArgs e)
		{
			ExecutionTimerEventHandler handler = ResultsPrinted;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		/// <summary>
		/// Raises the SeriesResultsPrinting event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnSeriesResultsPrinting(ExecutionTimerEventArgs e)
		{
			ExecutionTimerEventHandler handler = SeriesResultsPrinting;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		/// <summary>
		/// Raises the SeriesResultsPrinted event.
		/// </summary>
		/// <param name="e"><see cref="ExecutionTimerEventArgs"/> object containing
		/// the name of the <see cref="ExecutionTimer"/> firing this event.</param>
		protected virtual void OnSeriesResultsPrinted(ExecutionTimerEventArgs e)
		{
			ExecutionTimerEventHandler handler = SeriesResultsPrinted;

			if(handler != null)
			{
				handler(this, e);	
			}
		}

		#endregion
	
		#region IDisposable Members

		/// <summary>
		/// Causes Stop to be called.
		/// </summary>
		/// <remarks>Note: This method is not intended to be called from an 
		/// instance of ExecutionTimer, although doing so will not cause any 
		/// harm.</remarks>
		public void Dispose()
		{
			Stop();
		}

		#endregion
	}

	#region Delegates

	/// <summary>
	/// Represents the method called when output is being written.
	/// </summary>
	public delegate string PrintString(string timerName);

	/// <summary>
	/// Represents the method that handles the <see cref="ExecutionTimer.TimerStarting"/>, 
	/// <see cref="ExecutionTimer.ResultsPrinting"/>, 
	/// <see cref="ExecutionTimer.ResultsPrinted"/>, <see cref="ExecutionTimer.SeriesResultsPrinting"/>, 
	/// <see cref="ExecutionTimer.SeriesResultsPrinted"/> events of an <see cref="ExecutionTimer"/> 
	/// object.
	/// </summary>
	public delegate void ExecutionTimerEventHandler(object sender, ExecutionTimerEventArgs e);

	/// <summary>
	/// Represents the methods that handles the <see cref="ExecutionTimer.TimerStopped"/> 
	/// event of a <see cref="ExecutionTimer"/> object.
	/// </summary>
	public delegate void ExecutionTimerStoppedEventHandler(object sender, ExecutionTimerDetailedEventArgs e);

	#endregion

	#region EventArgs-derived classes

	/// <summary>
	/// Provides data for the <see cref="ExecutionTimer.TimerStarting"/>
	/// , <see cref="ExecutionTimer.ResultsPrinting"/>, <see cref="ExecutionTimer.ResultsPrinted"/>, 
	/// <see cref="ExecutionTimer.SeriesResultsPrinting"/>, <see cref="ExecutionTimer.SeriesResultsPrinted"/>
	///  events.
	/// </summary>
	public class ExecutionTimerEventArgs : EventArgs
	{
		private string _timerName;

		/// <summary>
		/// Initializes a new instance of the ExecutionTimerEventArgs class.
		/// </summary>
		/// <param name="timerName">The name of the <see cref="ExecutionTimer"/> that
		///  fired this event.</param>
		public ExecutionTimerEventArgs(string timerName)
		{
			_timerName = timerName;
		}

		/// <summary>
		/// Gets the name of the <see cref="ExecutionTimer"/> that fired this event.
		/// </summary>
		public string TimerName
		{
			get{ return _timerName; }
		}
	}

	/// <summary>
	/// Provides data for the <see cref="ExecutionTimer.TimerStopped"/> event.
	/// </summary>
	public class ExecutionTimerDetailedEventArgs : ExecutionTimerEventArgs
	{
		private TimeSpan _thisRun;
		private TimeSpan _totalSeries;
		private TimeSpan _shortestRun;
		private TimeSpan _longestRun;
		private int _runCount;

		/// <summary>
		/// Initializes a new instance of the <see cref="ExecutionTimerDetailedEventArgs"/>
		///  class with the specified timer name and run information.
		/// </summary>
		/// <param name="timerName">The name of the <see cref="ExecutionTimer"/>
		/// that fired this event.</param>
		/// <param name="thisRun">The execution time for this run.</param>
		/// <param name="runCount">The total number of runs in the series.</param>
		/// <param name="totalSeries">The total execution time for the series.</param>
		/// <param name="shortestRun">The execution time for the shortest run in the series.</param>
		/// <param name="longestRun">The execution time for the longest run in the series.</param>
		public ExecutionTimerDetailedEventArgs(string timerName, TimeSpan thisRun,
			int runCount, TimeSpan totalSeries, TimeSpan shortestRun, TimeSpan longestRun)
			: base (timerName)
		{
			_thisRun = thisRun;
			_runCount = runCount;
			_totalSeries = totalSeries;
			_shortestRun = shortestRun;
			_longestRun = longestRun;
		}

		/// <summary>
		/// Get the execution time for this run.
		/// </summary>
		public TimeSpan ThisRun
		{
			get{ return _thisRun; }
		}

		/// <summary>
		/// Gets the execution time for the entire series.
		/// </summary>
		public TimeSpan TotalSeries
		{
			get{ return _totalSeries; }
		}

		/// <summary>
		/// Gets the execution time for the shortest run in the series.
		/// </summary>
		public TimeSpan ShortestRun
		{
			get{ return _shortestRun; }
		}

		/// <summary>
		/// Gets the execution time for the longest run in the series.
		/// </summary>
		public TimeSpan LongestRun
		{
			get{ return _longestRun; }
		}

		/// <summary>
		/// Getst the number of runs executed in this series.
		/// </summary>
		public int RunCount
		{
			get{ return RunCount; }
		}
	}

	#endregion
}