using System;

namespace HarryBonesSolutions.Performance
{

	/// <summary>
	/// This class exists for the sole purpose of displaying the
	/// execution time, in milliseconds, of code contained in a 
	/// C# "using" block.
	/// </summary>
	/// <example>
	/// <code>
	/// void TakeOneSecond()
	/// {
	/// 	System.Threading.Thread.Sleep(1000);
	/// }
	/// 
	/// void RunTimer()
	/// {
	/// 	using(new BlockTimer("Timer for TakeOneSecond"))
	/// 	{
	/// 		TakeOneSecond();
	/// 	}
	/// }
	/// 
	/// /*
	/// 
	/// Output:
	/// 
	/// Timer for TakeOneSecond
	///  - Total Execution Time: 1000 ms.
	///  
	/// */ 
	/// </code>
	/// </example>
	public class BlockTimer : IDisposable
	{
		private string _description;
		private long	_start;

		/// <summary>
		/// Initializes a new instance of the <see cref="BlockTimer"/> class.
		/// </summary>
		/// <param name="description">The description to be printed with the results.</param>
		public BlockTimer(string description)
		{
			_description = description;
			_start = DateTime.Now.Ticks;
		}

		#region IDisposable Members

		/// <summary>
		/// Stops the timer and prints the results to the console.
		/// </summary>
		public void Dispose()
		{
			long totalTime = DateTime.Now.Ticks - _start;
			Console.WriteLine(_description);
			Console.Write(" - Total Execution Time: ");
			Console.Write(new TimeSpan(totalTime).TotalMilliseconds.ToString());
			Console.WriteLine(" ms.");
		}

		#endregion
	}
}