﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NeuronDotNet.Core;
using NeuronDotNet.Core.Backpropagation;
using NeuronDotNet.Core.Initializers;
using NeuronDotNet.Core.SOM;
using NeuronDotNet.Core.SOM.NeighborhoodFunctions;
using NUnit.Framework;
using Utils.Diagnostics;
using Utils.Log.Diagnostics;
using Uznavayko.NeyronNetwork.UznavaykoNetwork;
using Uznavayko.UznavaykoKeyboard.Service;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using UznavaykoNetwork;
using Assert = NUnit.Framework.Assert;
using Network = Uznavayko.NeyronNetwork.UznavaykoNetwork.Network;

namespace ServiceTests
{
  /// <summary>
  ///This is a test class for UznavaykoServiceTest and is intended
  ///to contain all UznavaykoServiceTest Unit Tests
  ///</summary>
  [TestFixture]
  public class UznavaykoServiceTest : LoggingTestFixture
  {
    #region private properties

    private TestContext testContextInstance;

    private UznavaykoService_Accessor target;

    /// <summary>
    /// Source for tracing events.
    /// </summary>
    private static readonly CyclicTraceSource log =
      TraceSourceFactory.CreateEntity();

    /// <summary>
    ///Gets or sets the test context which provides
    ///information about and functionality for the current test run.
    ///</summary>
    public TestContext TestContext
    {
      get
      {
        return testContextInstance;
      }
      set
      {
        testContextInstance = value;
      }
    }

    #endregion

    #region Additional test attributes

    //Use TestInitialize to run code before running each test
    [SetUp]
    public void MyTestInitialize()
    {
      //UznavaykoService_Accessor.paramsThatDoNotCount.Add(ParameterName.milliseconds);
      UznavaykoService_Accessor.paramsThatDoNotCount.Add(ParameterName.secondCode);
      UznavaykoService_Accessor.paramsThatDoNotCount.Add(ParameterName.firstCode);
      UznavaykoService_Accessor.paramsThatDoNotCount.Add(ParameterName.firstCodePressTime);

      target = new UznavaykoService_Accessor(
  "TestData\\KeyboardDataGoodSamples.xml",
  "TestData\\UznavaykoSecondCharTest2.dat",
  "TestData\\UznavaykoSecondCodeTest2.dat",
  "TestData\\UznavaykoMillisecondsTest2.dat",
  "TestData\\UznavaykoSecondTimeTest2.dat",
  "TestData\\UznavaykoFirstCharTest2.dat",
  "TestData\\UznavaykoFirstCodeTest2.dat",
  "TestData\\UznavaykoFirstTimeTest2.dat");
    }

    #endregion

    #region Private methods

    /// <summary>
    /// Gets the input vector schema.
    /// </summary>
    /// <param name="paramsThatDoNotCount">The params that do not count.</param>
    /// <param name="outputVector">The output vector.</param>
    /// <returns></returns>
    private IEnumerable<ParameterName> GetInputVectorSchema(
      IEnumerable<ParameterName> paramsThatDoNotCount,
      ParameterName outputVector)
    {
      return from param in
               Enum.GetValues(typeof(ParameterName)).OfType<ParameterName>()
             where param != outputVector && (paramsThatDoNotCount == null ||
             !paramsThatDoNotCount.Contains(param))
             select param;
    }

    /// <summary>
    /// Gets the results.
    /// </summary>
    /// <param name="samples">The samples.</param>
    /// <returns></returns>
    private double GetResults(Dictionary<ParameterName,
      IEnumerable<UznavaykoSample>> samples)
    {
      ICollection<double> results = new List<double>();
      foreach (var samplePair in samples)
      {
        double accuracy = 1;
        switch (samplePair.Key)
        {
          case ParameterName.firstChar:
            accuracy = 0.95;
            break;
          case ParameterName.firstCode:
            accuracy = 0.95;
            break;
          case ParameterName.firstCodePressTime:
            accuracy = 0.99;
            break;
          case ParameterName.hoursOfTheDay:
            accuracy = 0.95;
            break;
          case ParameterName.milliseconds:
            accuracy = 0.99;
            break;
          case ParameterName.secondChar:
            accuracy = 0.95;
            break;
          case ParameterName.secondCode:
            accuracy = 0.95;
            break;
          case ParameterName.secondCodePressTime:
            accuracy = 0.99;
            break;
        }

        TrainingSet set =
        Network.GetTrainingSet(samplePair.Value, 1000);

        float internalResult = GetResults(samplePair.Key, set, accuracy);
        results.Add(internalResult);

        log.Information("Mini results: {0}, {1}", samplePair.Key.ToString(),
          internalResult);
      }

      return results.Sum() / results.Count;
    }

    private float GetResults(ParameterName nameOfTheNetwork, TrainingSet trainingSet, double accuracy)
    {
      List<bool> results = new List<bool>();
      foreach (var sample in trainingSet.TrainingSamples)
      {
        double result = target.complexNetwork.
          networksHolder[nameOfTheNetwork].Run(sample.InputVector)[0];

        double waitingOuput = sample.OutputVector[0];

        if (waitingOuput != 0 && result != 0)
        {
          double tochnost = result / waitingOuput;
          results.Add(tochnost > accuracy && tochnost <= 1 ||
                   1 / tochnost > accuracy && 1 / tochnost <= 1);
        }
        else
        {
          results.Add(1 - accuracy > Math.Abs(result) && waitingOuput == 0 ||
                   1 - accuracy > Math.Abs(waitingOuput) && result == 0);
        }
      }
      float chislitel = 1;
      foreach (bool result in results)
      {
        if (result)
        {
          chislitel++;
        }
      }
      return chislitel / results.Count;
    }

    private void GetSamples(IDictionary<ParameterName,
      IEnumerable<UznavaykoSample>> samplesGood,
      string sampleFileName)
    {
      target.SamplesFileName = sampleFileName;
      foreach (ParameterName key in
        target.complexNetwork.networksHolder.Keys)
      {
        samplesGood.Add(key, target.GetSamples(key));
      }
    }

    private BackpropagationNetwork GetNetworkForSchemaTest(ParameterName parameter,
      out IEnumerable<UznavaykoSample> trainingSamples)
    {
      target.SamplesFileName = "TestData\\KeyboardDataGoodSamples.xml";
      const double learningRate = 0.3d;
      const int neuronCount = 10;

      trainingSamples =
        target.GetSamples(parameter);
      IList<Layers> networkSchena = target.complexNetwork.
        NetworkSchemesDictionary[parameter];

      LinearLayer inputLayer = new LinearLayer(6);
      ActivationLayer outputLayer =
        CreateNetwork(networkSchena, inputLayer, 1, neuronCount, learningRate);
      BackpropagationNetwork network = new BackpropagationNetwork(inputLayer, outputLayer);
      network.SetLearningRate(learningRate);
      return network;
    }

    private void AddLayerToSchema(IList<Layers> networkSchena, int place)
    {
      if (networkSchena.Count == 0)
      {
        networkSchena.Add(Layers.SineLayer);
        return;
      }

      if (place == -1)
      {
        networkSchena.Insert(0, Layers.SineLayer);
        return;
      }

      if ((int)networkSchena[place] + 1 >
        Enum.GetValues(typeof(Layers)).Length)
      {
        networkSchena[place] = Layers.SineLayer;
        AddLayerToSchema(networkSchena, place - 1);
        return;
      }

      networkSchena[place]++;
    }

    private string CreateNetworkSchemaLogString(
      List<Layers> networkSchena, float resultOfTheLearning)
    {
      StringBuilder builder = new StringBuilder();

      builder.Insert(builder.Length,
        String.Format("Result = {0}, Schema: \n LinearLayer(7), \n", resultOfTheLearning));

      foreach (Layers layers in networkSchena)
      {
        builder.Insert(builder.Length,
          Enum.GetName(typeof(Layers), layers) + ", \n");
      }

      return builder.ToString();
    }

    /// <summary>
    /// Creates the network by given schema.
    /// </summary>
    /// <param name="networkChema">The network chema.</param>
    /// <param name="inputLayer">The input layer.</param>
    /// <param name="numberOfTheOutputNeyrons">The number of the output neyrons.</param>
    /// <param name="neuronCount">The neuron count.</param>
    /// <param name="learningRate">The learning rate.</param>
    /// <returns>output layer</returns>
    private ActivationLayer CreateNetwork(IList<Layers> networkChema, ActivationLayer inputLayer,
      int numberOfTheOutputNeyrons, int neuronCount, double learningRate)
    {
      ActivationLayer layerToReturn = inputLayer;
      for (int i = 0; i < networkChema.Count; i++)
      {
        int numberOfNeyrons = i + 1 == networkChema.Count ?
          numberOfTheOutputNeyrons : neuronCount;

        switch (networkChema[i])
        {
          case Layers.LogarithmLayer:
            layerToReturn = CreateLayer(
              new LogarithmLayer(numberOfNeyrons),
              layerToReturn,
              learningRate);
            break;
          case Layers.SigmoidLayer:
            layerToReturn = CreateLayer(
              new SigmoidLayer(numberOfNeyrons),
              layerToReturn,
              learningRate);
            break;
          case Layers.SineLayer:
            layerToReturn = CreateLayer(
             new SineLayer(numberOfNeyrons),
             layerToReturn,
             learningRate);
            break;
          case Layers.TanhLayer:
            layerToReturn = CreateLayer(
             new TanhLayer(numberOfNeyrons),
             layerToReturn,
             learningRate);
            break;
        }
      }
      return layerToReturn;
    }

    /// <summary>
    /// Creates the layer.
    /// </summary>
    /// <param name="hiddenLayer">The hidden layer.</param>
    /// <param name="layerToReturn">The layer to return.</param>
    /// <param name="learningRate">The learning rate.</param>
    /// <returns></returns>
    private ActivationLayer CreateLayer(ActivationLayer hiddenLayer, ActivationLayer layerToReturn,
      double learningRate)
    {
      new BackpropagationConnector(layerToReturn, hiddenLayer).
        Initializer = new RandomFunction(0d, learningRate);

      layerToReturn = hiddenLayer;
      return layerToReturn;
    }

    #endregion

    #region Tests

    /// <summary>
    ///A test for GetCharResult
    ///</summary>
    [Test]
    public void GetCharResultTest()
    {
      target.NeedToLearnFromXml = true;
      target.LearnNetworks();

      Dictionary<ParameterName, IEnumerable<UznavaykoSample>> samplesBad =
        new Dictionary<ParameterName, IEnumerable<UznavaykoSample>>();

      GetSamples(samplesBad, "TestData\\KeyboardDataBadSamples.xml");

      double resultBad = GetResults(samplesBad);

      Dictionary<ParameterName, IEnumerable<UznavaykoSample>> samplesTestGood =
       new Dictionary<ParameterName, IEnumerable<UznavaykoSample>>();

      GetSamples(samplesTestGood, "TestData\\KeyboardDataGoodSamples.xml");

      double resultTestGood = GetResults(samplesTestGood);

      Assert.Greater(resultTestGood, resultBad);
    }

    [Test]
    public void FindBestNumberOfNeurons()
    {
      log.Information("Begin mega experiment");

      float resultOfTheLearning = 0;
      log.Information("loading samples from xml = TestData\\KeyboardDataGoodSamples.xml");
      target.SamplesFileName = "TestData\\KeyboardDataGoodSamples.xml";

      ParameterName nameOfTheOutput = ParameterName.secondChar;
      IEnumerable<UznavaykoSample> trainingSamples =
        target.GetSamples(nameOfTheOutput);

      double learningRate = 0.3d;
      int neuronCount = 1;
      log.Information("firstChar, learningRate = 0.3d, begin neuronCount = 1, Schema:LinearLayer(7),LogarithmLayer,LogarithmLayer,TanhLayer,SineLayer.");

      while (resultOfTheLearning < 1)
      {
        log.Information(String.Format("Result = {0} , neurons = {1}",
          resultOfTheLearning, neuronCount));

        BackpropagationNetwork network;
        LinearLayer inputLayer = new LinearLayer(7);
        LogarithmLayer hiddenLayer = new LogarithmLayer(neuronCount);
        LogarithmLayer hiddenLayer1 = new LogarithmLayer(neuronCount);
        TanhLayer hiddenLayer2 = new TanhLayer(neuronCount);
        SineLayer outputLayer = new SineLayer(1);

        //connect layers
        new BackpropagationConnector(inputLayer, hiddenLayer).
          Initializer = new RandomFunction(0d, learningRate);
        new BackpropagationConnector(hiddenLayer, hiddenLayer1).
          Initializer = new RandomFunction(0d, learningRate);
        new BackpropagationConnector(hiddenLayer1, hiddenLayer2).
          Initializer = new RandomFunction(0d, learningRate);
        new BackpropagationConnector(hiddenLayer2, outputLayer).
          Initializer = new RandomFunction(0d, learningRate);

        network = new BackpropagationNetwork(inputLayer, outputLayer);
        network.SetLearningRate(learningRate);
        resultOfTheLearning =
          new Network(nameOfTheOutput, network).Learn(trainingSamples, 100, 1000);
        neuronCount++;
      }
    }

    [Test]
    public void FindBestNetworkSchema()
    {
      log.Information("Begin mega experiment");

      float resultOfTheLearning = 0;
      log.Information("loading samples from xml = TestData\\KeyboardDataGoodSamples.xml");
      target.SamplesFileName = "TestData\\KeyboardDataGoodSamples.xml";

      ParameterName nameOfTheOutput = ParameterName.secondCodePressTime;
      IEnumerable<UznavaykoSample> trainingSamples =
        target.GetSamples(nameOfTheOutput);

      List<Layers> networkSchena = new List<Layers>();
      double learningRate = 0.3d;
      int neuronCount = 5;
      log.Information("learningRate = 0.3d, neuronCount = 5");
      while (resultOfTheLearning < 1)
      {
        log.Information(CreateNetworkSchemaLogString(
          networkSchena, resultOfTheLearning));

        BackpropagationNetwork network;
        LinearLayer inputLayer = new LinearLayer(7);
        ActivationLayer outputLayer;

        AddLayerToSchema(networkSchena, networkSchena.Count - 1);

        outputLayer = CreateNetwork(networkSchena,
          inputLayer, 1, neuronCount, learningRate);

        network = new BackpropagationNetwork(inputLayer, outputLayer);
        network.SetLearningRate(learningRate);
        resultOfTheLearning =
          new Network(nameOfTheOutput,network).Learn(trainingSamples, 100, 1000);
      }
    }

    [Test]
    public void NetworkSchemaTestFirstChar()
    {
      IEnumerable<UznavaykoSample> trainingSamples;
      ParameterName nameOfTheOutput = ParameterName.firstChar;
      BackpropagationNetwork network =
        GetNetworkForSchemaTest(nameOfTheOutput, out trainingSamples);

      UznavaykoSample.InputVectorSchema =
        new List<ParameterName> { 
            //ParameterName.firstChar, 
            ParameterName.firstCode,
            ParameterName.firstCodePressTime,
            ParameterName.milliseconds,
            ParameterName.secondChar,
            ParameterName.secondCode,
            ParameterName.secondCodePressTime};

      float resultOfTheLearning =
        new Network(nameOfTheOutput,network).Learn(trainingSamples, 100, 1000);
      Assert.Greater(resultOfTheLearning, 0.72);
    }

    [Test]
    public void NetworkSchemaTestFirstCode()
    {
      IEnumerable<UznavaykoSample> trainingSamples;
      ParameterName nameOfTheOutput = ParameterName.firstCode;
      BackpropagationNetwork network =
        GetNetworkForSchemaTest(nameOfTheOutput, out trainingSamples);

      UznavaykoSample.InputVectorSchema =
        new List<ParameterName> { 
            ParameterName.firstChar, 
            //ParameterName.firstCode,
            ParameterName.firstCodePressTime,
            ParameterName.milliseconds,
            ParameterName.secondChar,
            ParameterName.secondCode,
            ParameterName.secondCodePressTime};

      float resultOfTheLearning =
        new Network(nameOfTheOutput,network).Learn(trainingSamples, 100, 1000);
      Assert.Greater(resultOfTheLearning, 0.54);
    }

    [Test]
    public void NetworkSchemaTestSecondCodePressTime()
    {
      IEnumerable<UznavaykoSample> trainingSamples;
      ParameterName nameOfTheOutput = ParameterName.secondCodePressTime;
      BackpropagationNetwork network =
        GetNetworkForSchemaTest(nameOfTheOutput, out trainingSamples);

      UznavaykoSample.InputVectorSchema =
        new List<ParameterName> {
          ParameterName.firstChar, 
          ParameterName.firstCode,
          ParameterName.firstCodePressTime,
          ParameterName.milliseconds,
          ParameterName.secondChar,
          ParameterName.secondCode};

      float resultOfTheLearning =
        new Network(nameOfTheOutput,network).Learn(trainingSamples, 100, 1000);
      Assert.Greater(resultOfTheLearning, 0.36);
    }

    [Test]
    public void NetworkSchemaTestKohonen1()
    {
      const double learningRate = 0.3d;

      target.SamplesFileName = "TestData\\KeyboardDataGoodSamples.xml";
      ParameterName nameOfTheOutput = ParameterName.firstChar;
      IEnumerable<UznavaykoSample> trainingSamples = target.GetSamples(nameOfTheOutput);

      LinearLayer inputLayer = new LinearLayer(6);
      KohonenLayer hiddenLayer = new KohonenLayer(100);
      KohonenLayer hiddenLayer1 = new KohonenLayer(100);
      KohonenLayer hiddenLayer2 = new KohonenLayer(100);
      KohonenLayer hiddenLayer3 = new KohonenLayer(100);
      KohonenLayer outputLayer = new KohonenLayer(1);

      new KohonenConnector(inputLayer, hiddenLayer).
        Initializer = new RandomFunction();

      new KohonenConnector(hiddenLayer, hiddenLayer1).
        Initializer = new RandomFunction();

      new KohonenConnector(hiddenLayer1, hiddenLayer2).
       Initializer = new RandomFunction();

      new KohonenConnector(hiddenLayer2, hiddenLayer3).
        Initializer = new RandomFunction();

      new KohonenConnector(hiddenLayer3, outputLayer).
        Initializer = new RandomFunction();

      KohonenNetwork network = new KohonenNetwork(inputLayer, outputLayer);

      network.SetLearningRate(learningRate);

      UznavaykoSample.InputVectorSchema =
        new List<ParameterName> {
          //ParameterName.firstChar, 
          ParameterName.firstCode,
          ParameterName.firstCodePressTime,
          ParameterName.milliseconds,
          ParameterName.secondChar,
          ParameterName.secondCode,
          ParameterName.secondCodePressTime};

      float resultOfTheLearning =
        new Network(nameOfTheOutput,network).Learn(trainingSamples, 100, 1000);
      Assert.Greater(resultOfTheLearning, 0.36);
    }

    [Test]
    public void FindBestNetworkSchemaWithTochnost()
    {
      log.Information("Begin mega experiment");

      float resultOfTheLearning = 0;

      log.Information(
        "loading samples from xml = TestData\\KeyboardDataGoodSamples.xml");

      target.SamplesFileName = "TestData\\KeyboardDataGoodSamples.xml";
      ParameterName nameOfTheOutput = ParameterName.secondChar;
      IEnumerable<UznavaykoSample> trainingSamples =
        target.GetSamples(nameOfTheOutput);

      List<Layers> networkSchena = new List<Layers>();

      const double learningRate = 0.3d;
      const int neuronCount = 10;

      log.Information("learningRate = 0.3d, neuronCount = 10");
      while (resultOfTheLearning < 1)
      {
        BackpropagationNetwork network;
        LinearLayer inputLayer = new LinearLayer(7);
        ActivationLayer outputLayer;

        AddLayerToSchema(networkSchena, networkSchena.Count - 1);

        outputLayer = CreateNetwork(networkSchena,
          inputLayer, 1, neuronCount, learningRate);

        network = new BackpropagationNetwork(inputLayer, outputLayer);
        network.SetLearningRate(learningRate);

        resultOfTheLearning =
          new Network(nameOfTheOutput, network).Learn(trainingSamples, 1000, 1000);

        log.Information(CreateNetworkSchemaLogString(
          networkSchena, resultOfTheLearning));
      }
    }
    #endregion
  }
}
