using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;

namespace MouseAnalyzer.SecurityAlgorithms
{
    [Serializable()]
    class TraitsTree
    {
        private const double DifferenceCoefficientMax = 0.1;
        //      .
        private const int NodesCountMax = 10;

        private double _nodeValue;

        public double NodeValue
        {
            get { return _nodeValue; }
        } 

        private List<TraitsTree> _nodes;

        public TraitsTree()
        {
            _nodes = new List<TraitsTree>();
            _nodeValue = 0;
        }


        public void AddTraitsVector(double[] vector)
        {
            if (vector.Length == 0)
            {
                throw new ArgumentException();
            }

            AddTraitsVector(vector, 0);
        }

        private void AddTraitsVector(double[] vector, int currentElement)
        {
            if (currentElement < vector.Length)
            {
                TraitsTree currentElementNearestNode = FindCorrespondingNode(vector[currentElement]);
                if (currentElementNearestNode != null)
                {
                    currentElementNearestNode._nodeValue = (currentElementNearestNode.NodeValue + vector[currentElement]) / 2;
                }
                else
                {
                    currentElementNearestNode = new TraitsTree();
                    currentElementNearestNode._nodeValue = vector[currentElement];
                    _nodes.Add(currentElementNearestNode);
                }
                currentElementNearestNode.AddTraitsVector(vector, currentElement + 1);
            }
        }

        public double CheckTraitsVector(double[] vector)
        {
            if (vector.Length == 0)
            {
                throw new ArgumentException();
            }

            return 1 - CheckTraitsVector(vector, 0) / vector.Length;
        }

        private double CheckTraitsVector(double[] vector, int currentElement)
        {
            if (currentElement < vector.Length)
            {
                TraitsTree currentElementNearestNode = FindCorrespondingNode(vector[currentElement]);
                if (currentElementNearestNode != null)
                {
                    return CalculateRelativeDifference(currentElementNearestNode.NodeValue, vector[currentElement]) + CheckTraitsVector(vector, currentElement + 1);
                }
                else
                {
                    return 0;
                }
            }
            return 0;
        }

        private TraitsTree FindCorrespondingNode(double value)
        {
            if (_nodes.Count > 0)
            {
                TraitsTree minDifferenceElement = _nodes[0];
                double minDifference = Math.Abs(_nodes[0].NodeValue - value);
                for (int i = 1; i < _nodes.Count; i++)
                {
                    double difference = Math.Abs(_nodes[i].NodeValue - value);
                    if (minDifference > difference)
                    {
                        minDifference = difference;
                        minDifferenceElement = _nodes[i];
                    }
                }
                // ,      10%      
                if ((minDifference < value * DifferenceCoefficientMax)
                    || (_nodes.Count >= NodesCountMax))
                {
                    return minDifferenceElement;
                }
                else
                {
                    return null;
                }
            }
            else
            {
                return null;
            }
        }

        private void UpdateNodeValue(double nodeValue)
        {
            if (Math.Abs(_nodeValue - nodeValue) < _nodeValue * DifferenceCoefficientMax)
            {
                _nodeValue = (nodeValue + _nodeValue) / 2;
            }
        }

        private double CalculateRelativeDifference(double nodeValue, double currentValue)
        {
            //     
            if (nodeValue == 0)
            {
                return 0;
            }
            return Math.Abs((nodeValue - currentValue) / nodeValue);

        }

        public void Save(string fileName)
        {
            using (FileStream fs = new FileStream(fileName, FileMode.OpenOrCreate))
            {
                BinaryFormatter binFormatter = new BinaryFormatter();
                binFormatter.Serialize(fs, this);
            }
        }

        public static TraitsTree Load(string fileName)
        {
            FileStream fs = new FileStream(fileName, FileMode.OpenOrCreate);
            BinaryFormatter binFormatter = new BinaryFormatter();
            TraitsTree tree = (TraitsTree)binFormatter.Deserialize(fs);
            fs.Close();
            return tree;
        }
    }
}
