using System;
using System.Collections.Generic;
using System.Text;
using MouseAnalyzer.MouseTraits;

namespace MouseAnalyzer.Managers
{

    public delegate void TraitVectorAvailableEventHandler(object sender, TraitVectorAvailableEventArgs e);
    public delegate void TraitAddedEventHandler(object sender, TraitAddedEventArgs e);

    class TraitsManager
    {

        public event TraitVectorAvailableEventHandler NewTraitsVectorAvailable;
        public event TraitAddedEventHandler TraitAdded;

        private const int MovementTrajectoryPointsCountMin = 10;
        private const int WheelScrollingsCountMin = 2;
        private const int WheelScrollingDurationMax = 500;

        private const double Epsilon = 1E-10;

        // TODO:       ...
        private const int TraitsCountInListMax = 1000;
        private const int TraitsToRemoveCountIfMaxReached = TraitsCountInListMax / 2;

        private Dictionary<MouseTraitType, List<ITrait>> _mouseTraits;

        private int _traitsUsedCount;

        private static TraitsManager _instance;

        // TODO:       ?

        private TraitsManager()
        {
            _mouseTraits = new Dictionary<MouseTraitType, List<ITrait>>();
            Array mouseTraitsKeys = Enum.GetValues(typeof(MouseTraitType));
            Array.Sort(mouseTraitsKeys);
            foreach (int key in mouseTraitsKeys)
            {
                _mouseTraits.Add((MouseTraitType)key, new List<ITrait>(TraitsCountInListMax));
            }
            _traitsUsedCount = 0;
        }

        public static TraitsManager GetIntstance()
        {
            if (_instance == null)
            {
                _instance = new TraitsManager();
            }
            return _instance;
        }

        private bool CheckMovementTrait(MovementMouseTrait movementMouseTrait)
        {
            return (movementMouseTrait.TrajectoryPointsCount > MovementTrajectoryPointsCountMin);
        }

        public bool CheckWheelScrollingTrait(WheelScrollingMouseTrait scrollingTrait)
        {
            return (scrollingTrait.Duration < WheelScrollingDurationMax)
                && (scrollingTrait.ScrollingsCount > WheelScrollingsCountMin);
        }

        public bool AddNewTrait(ITrait newTrait, MouseTraitType traitType)
        {
            if (CheckTrait(newTrait, traitType))
            {
                if (_mouseTraits.ContainsKey(traitType))
                {
                    _mouseTraits[traitType].Add(newTrait);
                    OnTraitAdded(this, new TraitAddedEventArgs(traitType));
                    GenerateNewTraitVectorEventIfNeeded();
                    return true;
                }
            }
            return false;
        }

        private bool CheckTrait(ITrait newTrait, MouseTraitType traitType)
        {
            if (traitType == MouseTraitType.Movement)
            {
                MovementMouseTrait movementTrait = (MovementMouseTrait)newTrait;
                if (movementTrait.TrajectoryPointsCount < MovementTrajectoryPointsCountMin)
                {
                    return false;
                }
            }
            else if (traitType == MouseTraitType.WheelScrolling)
            {
                WheelScrollingMouseTrait wheelTrait = (WheelScrollingMouseTrait)newTrait;
                if ((wheelTrait.Duration > WheelScrollingDurationMax)
                    || (wheelTrait.ScrollingsCount < WheelScrollingsCountMin))
                {
                    return false;
                }
            }
            return true;
        }

        #region Events

        private void GenerateNewTraitVectorEventIfNeeded()
        {
            // ,          ...
            bool traitsAvailable = true;
            foreach (List<ITrait> lst in _mouseTraits.Values)
            {
                if (lst.Count == 0 || lst.Count <= _traitsUsedCount)
                {
                    traitsAvailable = false;
                    break;
                }
            }
            if (traitsAvailable)
            {
                // TODO: -    ...      ITrait  ...
                List<double> traitsList = new List<double>();
                foreach (List<ITrait> lst in _mouseTraits.Values)
                {
                    traitsList.AddRange(lst[_traitsUsedCount].GenerateTraitVector());
                }
                double[] traitsVector = traitsList.ToArray();
                if (CheckGeneratedVector(traitsVector))
                {
                    OnNewTraitsVectorAvailable(this, new TraitVectorAvailableEventArgs(traitsVector));
                }
                _traitsUsedCount++;
            }
        }

        private bool CheckGeneratedVector(double[] vector)
        {
            for (int i = 0; i < vector.Length; i++)
            {
                if (vector[i] < Epsilon)
                {
                    return false;
                }
            }
            return true;
        }

        private void OnNewTraitsVectorAvailable(object sender, TraitVectorAvailableEventArgs e)
        {
            TraitVectorAvailableEventHandler tempEventHandler = NewTraitsVectorAvailable;
            if (tempEventHandler != null)
            {
                tempEventHandler(sender, e);
            }
        }

        private void OnTraitAdded(object sender, TraitAddedEventArgs e)
        {
            TraitAddedEventHandler tempEventHandler = TraitAdded;
            if (tempEventHandler != null)
            {
                tempEventHandler(sender, e);
            }
        }
        #endregion
    }
}
