using System;
using System.Collections.Generic;
using System.Text;

namespace MouseAnalyzer
{
    class CubicSpline
    {
        // 
        private double[] _aX;
        private double[] _bX;
        private double[] _cX;
        private double[] _dX;

        private double[] _aY;
        private double[] _bY;
        private double[] _cY;
        private double[] _dY;

        public int NodesCount
        {
            get
            {
                return _aX.Length;
            }
        }

        private CubicSpline(double[] aX, double[] bX, double[] cX, double[] dX,
                            double[] aY, double[] bY, double[] cY, double[] dY)
        {
            _aX = aX;
            _bX = bX;
            _cX = cX;
            _dX = dX;

            _aY = aY;
            _bY = bY;
            _cY = cY;
            _dY = dY;
        }

        public double CalculateCurvature(int pointIndex)
        {
            if (pointIndex < 0 || pointIndex > NodesCount)
            {
                throw new ArgumentException();
            }
            double tmp = _bX[pointIndex] * _bX[pointIndex] + _bY[pointIndex] * _bY[pointIndex];
            tmp = Math.Sqrt(tmp * tmp * tmp);

            return (2 * Math.Abs(_cY[pointIndex] * _bX[pointIndex] - _cX[pointIndex] * _bY[pointIndex]) / tmp);
        }

        public static CubicSpline InterpolatePolyline(double[] x, double[] y)
        {
            double[] parameterValues = new double[x.Length];
            double currentLength = 0;
            for (int pointIndex = 0; pointIndex < x.Length - 1; pointIndex++)
            {
                parameterValues[pointIndex] = currentLength;
                currentLength += Math.Sqrt(x[pointIndex] * x[pointIndex] + y[pointIndex] * y[pointIndex]);
            }
            parameterValues[x.Length - 1] = currentLength;

            double[] aX = new double[x.Length];
            double[] bX = new double[x.Length];
            double[] cX = new double[x.Length];
            double[] dX = new double[x.Length];
            InterpolateFunction(parameterValues, x, ref aX, ref bX, ref cX, ref dX);

            double[] aY = new double[x.Length];
            double[] bY = new double[x.Length];
            double[] cY = new double[x.Length];
            double[] dY = new double[x.Length];
            InterpolateFunction(parameterValues, y, ref aY, ref bY, ref cY, ref dY);

            return new CubicSpline(aX, bX, cX, dX, aY, bY, cY, dY);
        }

        private static void InterpolateFunction(
            double[] param, 
            double[] paramFunction,
            ref double[] aCoeffs,
            ref double[] bCoeffs, 
            ref double[] cCoeffs,  
            ref double[] dCoeffs)
        {
            for (int i = 0; i < param.Length; i++)
            {
                aCoeffs[i] = paramFunction[i];
            }

            double[] hi = new double[param.Length];
            for (int i = 0; i < param.Length - 1; i++)
            {
                hi[i] = param[i + 1] - param[i];
            }
            hi[param.Length - 1] = hi[0];
            double[] gVector = new double[param.Length];
            for (int i = 1; i < param.Length - 1; i++)
            {
                gVector[i] = (3 / hi[i]) * (aCoeffs[i + 1] - aCoeffs[i]) - (3 / hi[i - 1]) * (aCoeffs[i] - aCoeffs[i - 1]);
            }
            gVector[param.Length - 1] = (3 / hi[param.Length - 1]) * (aCoeffs[1] - aCoeffs[0]) - (3 / hi[param.Length - 2]) * (aCoeffs[param.Length - 1] - aCoeffs[param.Length - 2]);

            if (param.Length > 3)
            {
                double[] alfa = new double[param.Length];
                double[] beta = new double[param.Length];
                double[] gamma = new double[param.Length];
                double[] delta = new double[param.Length];

                alfa[1] = 2 * (hi[0] + hi[1]);
                gamma[1] = hi[1] / alfa[1];
                delta[1] = hi[0] / alfa[1];
                for (int i = 2; i < param.Length - 2; i++)
                {
                    alfa[i] = 2 * (hi[i - 1] + hi[i]) - hi[i - 1] * gamma[i - 1];
                    gamma[i] = hi[i] / alfa[i];
                    beta[i] = hi[i - 1];
                    delta[i] = -beta[i] * delta[i - 1] / alfa[i];
                }

                alfa[param.Length - 2] = 2 * (hi[param.Length - 3] + hi[param.Length - 2]) - hi[param.Length - 3] * gamma[param.Length - 3];
                beta[param.Length - 2] = hi[param.Length - 3];

                double[] epsilon = new double[param.Length];
                epsilon[3] = hi[0];
                for (int i = 4; i < param.Length; i++)
                {
                    epsilon[i] = -epsilon[i - 1] * gamma[i - 3];
                }

                gamma[param.Length - 2] = (hi[param.Length - 2] - beta[param.Length - 2] * delta[param.Length - 3]) / alfa[param.Length - 2];
                beta[param.Length - 1] = hi[param.Length - 2] - epsilon[param.Length - 1] * gamma[param.Length - 3];
                double sum = 0;
                for (int i = 3; i < param.Length; i++)
                {
                    sum += epsilon[i] * delta[i - 2];
                }
                alfa[param.Length - 1] = 2 * (hi[param.Length - 2] + hi[param.Length - 1]) - sum - beta[param.Length - 1] * gamma[param.Length - 2];

                double[] ci = new double[param.Length];
                ci[1] = gVector[1] / alfa[1];
                for (int i = 2; i < param.Length - 1; i++)
                {
                    ci[i] = (gVector[i] - ci[i - 1] * beta[i]) / alfa[i];
                }
                sum = 0;
                for (int i = 3; i < param.Length; i++)
                {
                    sum += epsilon[i] * ci[i - 2];
                }
                ci[param.Length - 1] = (gVector[param.Length - 1] - sum - beta[param.Length - 1] * ci[param.Length - 2]) / alfa[param.Length - 1];

                cCoeffs[param.Length - 1] = ci[param.Length - 1];
                cCoeffs[param.Length - 2] = ci[param.Length - 2] - gamma[param.Length - 2] * cCoeffs[param.Length - 1];
                for (int i = param.Length - 3; i > 0; i--)
                {
                    cCoeffs[i] = ci[i] - gamma[i] * cCoeffs[i + 1] - delta[i] * cCoeffs[param.Length - 1];
                }
                cCoeffs[0] = cCoeffs[param.Length - 1];
            }
            else if (param.Length == 3)
            {
                for (int i = 1; i < param.Length; i++)
                {
                    cCoeffs[i] = (2 * gVector[i] - gVector[param.Length - i]) / (3 * (hi[0] + hi[1]));
                }
                cCoeffs[0] = cCoeffs[param.Length - 1];
            }

            for (int i = 0; i < param.Length - 1; i++)
            {
                dCoeffs[i] = (1 / (3 * hi[i])) * (cCoeffs[i + 1] - cCoeffs[i]);
                bCoeffs[i] = (1 / hi[i]) * (aCoeffs[i + 1] - aCoeffs[i]) - (hi[i] / 3) * (cCoeffs[i + 1] + 2 * cCoeffs[i]);
            }
            bCoeffs[param.Length - 1] = bCoeffs[0];
            dCoeffs[param.Length - 1] = dCoeffs[0];

        }
    }
}
