﻿using System;
using System.Linq;
using System.Threading;
using UznavaykoNetwork;
using Utils.Diagnostics;
using System.ServiceModel;
using System.Configuration;
using System.ServiceProcess;
using Utils.Log.Diagnostics;
using System.Collections.Generic;
using Uznavayko.NeyronNetwork.UznavaykoNetwork;

namespace Uznavayko.UznavaykoKeyboard.Service
{
  /// <summary>
  /// Service that realize the work of uznavayko.
  /// </summary>
  public class UznavaykoService : ServiceBase
  {
    #region private fields

    private int validationErrors = 1;
    private int validationNumber = 1;

    /// <summary>
    /// 
    /// </summary>
    private static List<ParameterName> paramsThatDoNotCount =
      new List<ParameterName> { ParameterName.hoursOfTheDay };

    /// <summary>
    /// Server that provides info about Uznavaykos work via WCF.
    /// </summary>
    private readonly StateProvider service;

    /// <summary>
    /// Instance of the NetPipe service.
    /// </summary>
    private readonly ServiceHost host;

    /// <summary>
    /// Source for tracing events.
    /// </summary>
    private static readonly CyclicTraceSource log =
      TraceSourceFactory.CreateEntity();

    /// <summary>
    /// Contains Networks.
    /// </summary>
    private readonly ComplexNetwork complexNetwork =
      new ComplexNetwork(ouyputDelimiter);

    /// <summary>
    /// This is delimiter of the output vector.
    /// </summary>
    private const int ouyputDelimiter = 1000;

    /// <summary>
    /// Collects data for neuron network. Intercepts user input, 
    /// and can save in xml.
    /// </summary>
    private readonly DataCollector.DataCollector dataCollector;

    #endregion

    #region private methods

    /// <summary>
    /// Learns the networks.
    /// </summary>
    private void LearnNetworks()
    {
      Dictionary<ParameterName, IEnumerable<UznavaykoSample>> samples =
        new Dictionary<ParameterName, IEnumerable<UznavaykoSample>>();

      foreach (ParameterName key in complexNetwork.networksHolder.Keys)
      {
        samples.Add(key, GetSamples(key));
      }
      //this is needed for work of the windows hooks
      //thats why we read samples from xml even if do not need to
      if (!NeedToLearnFromXml)
      {
        return;
      }

      complexNetwork.LearnNetworks(samples);
    }

    /// <summary>
    /// Using XmlDataExposer class loads samples from xml file and 
    /// initializes two dictionary samplesFirstChar and samplesSecondChar
    /// that will be used to train networks.
    /// </summary>
    private IEnumerable<UznavaykoSample> GetSamples(
      ParameterName parameterName)
    {
      IList<UznavaykoSample> samples =
        XmlDataExposer.GetSamples(SamplesFileName, parameterName);

      for (int i = 0; i < samples.Count; i++)
      {
        yield return samples[i];
      }
    }

    /// <summary>
    /// Launches the operation in separate thread.
    /// </summary>
    /// <param name="action">The action.</param>
    /// <param name="actionParametaer">The action parameter.</param>
    private static void LaunchOperationInSeparateThread(
      Action<object> action,
      object actionParametaer)
    {
      Thread thread = new Thread(new ParameterizedThreadStart(action));
      thread.SetApartmentState(ApartmentState.STA);
      thread.IsBackground = true;
      thread.Start(actionParametaer);
    }

    private Dictionary<ParameterName, double> pairOld;
    /// <summary>
    /// Handles DataCollector key pressed event.
    /// If user is not recognized then learn on him if LearningMode is on.
    /// </summary>
    /// <param name="sender">The sender.</param>
    /// <param name="e">The <see cref="System.EventArgs"/>
    /// instance containing the event data.</param>
    private void DataCollectorKeyPressed(object sender, EventArgs e)
    {
      var pairNew = (Dictionary<ParameterName, double>)sender;

      if (pairOld == null)
      {
        pairOld = pairNew;
        return;
      }
      validationNumber++;
      List<bool> resultCounter = new List<bool>();
      string results = "Results:";
      foreach (ParameterName key in complexNetwork.networksHolder.Keys)
      {
        UznavaykoSample sampleOld = new UznavaykoSample(key, pairOld);
        UznavaykoSample sampleNew = new UznavaykoSample(key, pairNew);
        bool innerResult =
         complexNetwork.GetCharResult(key, sampleOld, sampleNew, Accuracy[key]);

        resultCounter.Add(innerResult);

        results += String.Format("key ={0}, Result={1}; ",
          key, innerResult);
      }

      pairOld = pairNew;

      log.Information(results);

      if (!NotifyAboutErrors)
      {
        return;
      }

      if (!resultCounter.Contains(false))
      {
        log.Information("Network reports success with data: firstChar={0}, firstCode={1},firstCodePressTime={2}, hoursOfTheDay={3}, milliseconds={4}, secondChar={5}, secondCode={6}, secondCodePressTime={7}",
        pairOld[ParameterName.firstChar],
        pairOld[ParameterName.firstCode],
        pairOld[ParameterName.firstCodePressTime],
        pairOld[ParameterName.hoursOfTheDay],
        pairOld[ParameterName.milliseconds],
        pairOld[ParameterName.secondChar],
        pairOld[ParameterName.secondCode],
        pairOld[ParameterName.secondCodePressTime]);
        service.NoErrors = true;
        return;
      }

      validationErrors++;

      log.Error("Network reports error with data: firstChar={0}, firstCode={1},firstCodePressTime={2}, hoursOfTheDay={3}, milliseconds={4}, secondChar={5}, secondCode={6}, secondCodePressTime={7}",
        pairOld[ParameterName.firstChar],
        pairOld[ParameterName.firstCode],
        pairOld[ParameterName.firstCodePressTime],
        pairOld[ParameterName.hoursOfTheDay],
        pairOld[ParameterName.milliseconds],
        pairOld[ParameterName.secondChar],
        pairOld[ParameterName.secondCode],
        pairOld[ParameterName.secondCodePressTime]);
      service.NoErrors = false;
    }

    /// <summary>
    /// Starts the data collector.
    /// </summary>
    /// <param name="o">The o.</param>
    private void StartDataCollector(object o)
    {
      dataCollector.KeyPressed += DataCollectorKeyPressed;
      dataCollector.StartCollector();
    }

    #endregion

    #region protedted method

    /// <summary>
    /// When implemented in a derived class, executes when a Start command is sent to the service by the Service Control Manager (SCM) or when the operating system starts (for a service that starts automatically). Specifies actions to take when the service starts.
    /// </summary>
    /// <param name="args">Data passed by the start command.</param>
    protected override void OnStart(string[] args)
    {
      log.Information("Start");

      host.Open();
    }

    /// <summary>
    /// </summary>
    public void OnStart()
    {
      OnStart(new[] { "learn" });
    }

    /// <summary>
    /// When implemented in a derived class, executes when a Stop command is sent to the service by the Service Control Manager (SCM). Specifies actions to take when a service stops running.
    /// </summary>
    protected override void OnStop()
    {
      Save();
    }

    #endregion

    #region public events and properties

    /// <summary>
    /// Gets or sets the accuracy.
    /// </summary>
    /// <value>The accuracy.</value>
    public Dictionary<ParameterName, double> Accuracy { get; set; }

    /// <summary>
    /// Gets or sets a value indicating whether [notify about errors].
    /// </summary>
    /// <value><c>true</c> if [notify about errors]; otherwise, <c>false</c>.</value>
    public bool NotifyAboutErrors { get; set; }

    /// <summary>
    /// Gets or sets a value indicating whether service will save collected data
    /// in XML file or only in data file.
    /// </summary>
    /// <value><c>true</c> saving data in XML and DAT files, 
    /// <c>false</c> saving only in DAT files.</value>
    public bool NeedToSaveInXml { get; set; }

    /// <summary>
    /// Gets or sets a value indicating whether service will learn neyron network
    /// with XML file or only will load the data files.
    /// </summary>
    public bool NeedToLearnFromXml { get; set; }

    /// <summary>
    /// Samples file name.
    /// </summary>
    public string SamplesFileName { get; set; }

    #endregion

    #region public methods

    /// <summary>
    /// Saves this instance to dat file.
    /// </summary>
    public void Save()
    {
      log.Information("Session statistics: {0}.",
        (double)validationErrors / validationNumber);

      if (NeedToSaveInXml)
      {
        dataCollector.SaveData();
      }
      log.Information("Saving network.");
      complexNetwork.Save();
      log.Information("Saving completed.");
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="UznavaykoService"/> class.
    /// Try to get parameters from config if no 
    /// config found then 
    /// initializes public properties with default values:
    /// LearningMode = true,
    /// NotifyAboutErrors = true,
    /// SamplesFileName = "KeyboardData.xml",
    /// NeuronSecondCharFileName = "UznavaykoSecondChar.dat",
    /// NeuronFirstCharFileName = "UznavaykoFirstChar.dat",
    /// ServiceName = "UznavaykoService"
    /// </summary>
    public UznavaykoService()
      : this("KeyboardDataGoodSamples.xml",
      "UznavaykoSecondChar.dat",
      "UznavaykoSecondCode.dat",
      "UznavaykoSecondTime.dat",
      "UznavaykoMilliseconds.dat",
      "UznavaykoFirstChar.dat",
      "UznavaykoFirstCode.dat",
      "UznavaykoFirstTime.dat")
    {
    }

    /// <summary>
    /// Constructor.
    /// </summary>
    public UznavaykoService(string samplesFileName,
      string neuronSecondCharFileName,
      string neuronSecondCodeFileName,
      string neuronSecondTimeFileName,
      string neuronMilliseconds,
      string neuronFirstCharFileName,
      string neuronFirstCodeFileName,
      string neuronFirstTimeFileName)
    {
      validationErrors = 0;
      validationNumber = 0;
      UznavaykoServiceConfig config =
        ConfigurationManager.GetSection(
        typeof(UznavaykoService).Namespace) as
        UznavaykoServiceConfig;

      if (config != null)
      {
        complexNetwork.LearningMode = config.LearningMode;
        NotifyAboutErrors = config.NotifyAboutErrors;
        ServiceName = config.ServiceName;
        NeedToSaveInXml = config.NeedToSaveInXml;
        NeedToLearnFromXml = config.NeedToLearnFromXml;
      }
      else
      {
        complexNetwork.LearningMode = true;
        NotifyAboutErrors = true;
        ServiceName = "UznavaykoService";
      }

      SamplesFileName = samplesFileName;
      dataCollector = new DataCollector.DataCollector(SamplesFileName);
      LaunchOperationInSeparateThread(StartDataCollector, false);

      complexNetwork.NeuronSecondCharFileName = neuronSecondCharFileName;
      complexNetwork.NeuronSecondCodeFileName = neuronSecondCodeFileName;
      complexNetwork.NeuronSecondTimeFileName = neuronSecondTimeFileName;
      complexNetwork.NeuronMilliseconds = neuronMilliseconds;
      complexNetwork.NeuronFirstCharFileName = neuronFirstCharFileName;
      complexNetwork.NeuronFirstCodeFileName = neuronFirstCodeFileName;
      complexNetwork.NeuronFirstTimeFileName = neuronFirstTimeFileName;

      complexNetwork.CreateNetwork(paramsThatDoNotCount);
      Accuracy = new Dictionary<ParameterName, double>
                   {
                     {ParameterName.firstChar, 0.99},
                     {ParameterName.firstCode, 0.99},
                     {ParameterName.firstCodePressTime, 0.99},
                     {ParameterName.hoursOfTheDay, 0.99},
                     {ParameterName.milliseconds, 0.99},
                     {ParameterName.secondChar, 0.99},
                     {ParameterName.secondCode, 0.99},
                     {ParameterName.secondCodePressTime, 0.99}
                   };


      LearnNetworks();
      service = new StateProvider();
      host = new ServiceHost(service);
    }
    #endregion
  }
}
