using System;
using System.Diagnostics;
using System.Reflection;
using Utils.Log.Diagnostics;

namespace Utils.Diagnostics
{
  /// <summary>
  /// Static factory to generate trace sources. Links up the 
  /// shared listeners and shared switch to every trace source
  /// </summary>
  public static class TraceSourceFactory
  {
    #region private fields

    /// <summary>
    /// Name to initialize shared switch entity
    /// </summary>
    private const string sharedSwitchName = "SharedSourceSwitch";

    /// <summary>
    /// Name for template trace source
    /// </summary>
    private const string traceSourceName = "Uznavayko";

    /// <summary>
    /// Shared switch entity
    /// </summary>
    private static readonly SourceSwitch sharedSwitch;

    /// <summary>
    /// Template for trace source creating (resulting item 
    /// has equal listeners list and switch settings)
    /// </summary>
    private static readonly CyclicTraceSource templateTraceSource;

    /// <summary>
    /// Lock object is used to manage listeners collection.
    /// </summary>
    private static Object lockObject = new Object();


    #endregion

    #region private member functions

    /// <summary>
    /// Create source switch
    /// </summary>
    private static SourceSwitch CreateSharedSwitch()
    {
      SourceLevels srcLevel = templateTraceSource.Switch.Level;

      // Create switch entity
      SourceSwitch srcSwitch = new SourceSwitch(
        sharedSwitchName, srcLevel.ToString());

      return srcSwitch;
    }

    /// <summary>
    /// Static constructor
    /// </summary>
    static TraceSourceFactory()
    {
      // Initialize template trace source
      templateTraceSource = new CyclicTraceSource(traceSourceName);

      // Initialize shared source switch entity
      sharedSwitch = CreateSharedSwitch();
    }

    #endregion

    #region public member functions

    /// <summary>
    /// Create a trace source
    /// </summary>
    /// <returns>
    /// Trace source entity with assigned shared 
    /// switch and a list of shared listeners
    /// </returns>
    /// <remarks>Not to call from the current class</remarks>
    public static CyclicTraceSource CreateEntity()
    {
      // Get stack trace (first frame represents 
      // current class, so it sould be skipped)
      StackTrace trace = new StackTrace(1);

      // Get necessary stack frame 
      // (only the first frame is needed)
      StackFrame frame = trace.GetFrame(0);

      // Get calling method from stack frame
      MethodBase callingMethod = frame.GetMethod();

      // Type of the calling object
      string strType = callingMethod.DeclaringType.FullName;

      // Create trace source (result of this method)
      CyclicTraceSource traceSource = new CyclicTraceSource(strType);

      // Remove default trace listener
      if ((traceSource.Listeners.Count > 0) && (traceSource.Listeners[0] is DefaultTraceListener))
      {
        traceSource.Listeners.RemoveAt(0);
      }

      lock (lockObject)
      {
        // Assign shared listeners to the trace source
        traceSource.Listeners.AddRange(templateTraceSource.Listeners);
      }

      // Assign shared switch to the trace source
      traceSource.Switch = sharedSwitch;

      traceSource.Verbose("Trace source created for {0}", callingMethod.DeclaringType.Name);
      return traceSource;

    } // end CreateEntity

    /// <summary>
    /// Source level for the shared switch
    /// </summary>
    public static SourceLevels SwitchLevel
    {
      get { return sharedSwitch.Level; }
      set
      {
        Object lockObject = new Object();
        lock (lockObject)
        {
          sharedSwitch.Level = value;
        }
      }
    }

    /// <summary>
    /// This method is used to add trace listener to the collection of the template listeners.
    /// Used ainly for unit testing purposes (to add console listener manually).
    /// </summary>
    /// <param name="listener">Listener to add</param>
    internal static void AddTraceListener(TraceListener listener)
    {
      if(listener == null)
      {
        throw new ArgumentNullException("listener");
      }

      lock (lockObject)
      {
        // Check if we already have such listener
        if(templateTraceSource.Listeners.Contains(listener))
        {
          return;
        }

        // Add listener to the collection of template listeners
        templateTraceSource.Listeners.Add(listener);
      }
    }

    #endregion

  } // end class TraceSourceFactory
}