using System;
using System.IO;
using System.Collections;
using System.Runtime.InteropServices;
using mscoree;
using NUnit.Framework;
using System.Diagnostics;
using System.Reflection;

namespace Utils.Diagnostics
{
  /// <summary>
  /// This is a base class for all test fixtures with enabled logging.
  /// </summary>
  public abstract class LoggingTestFixture
  {
    #region private fields
    private XmlHelper xmlHelper;
    #endregion

    #region protected properties
    /// <summary>
    /// Returns full path to directory where current assembly is located.
    /// </summary>
    protected string AssmeblyFolder
    {
      get
      {
        string result = AppDomain.CurrentDomain.BaseDirectory;

        Assert.That(
          File.Exists(Path.Combine(result, Path.GetFileName(MyAssembly.Location))),
          "Could not find assembly {0} in directory {1}, assembly folder evaluation failed.",
          Path.GetFileName(MyAssembly.Location),
          result);

        return result;
      }
    }

    /// <summary>
    /// Returns an assembly of the current type.
    /// </summary>
    protected Assembly MyAssembly
    {
      get
      {
        return this.GetType().Assembly;
      }
    }

    /// <summary>
    /// Returns helper class to use for manipulations with xml.
    /// </summary>
    protected XmlHelper XmlHelper
    {
      get
      {
        if (xmlHelper == null)
        {
          Assert.Fail("Test fixture is not initialized yet.");
        }
        return xmlHelper;
      }
    }
    #endregion

    #region private methods

    private static void ListAppDomains()
    {        
      CorRuntimeHostClass host = new CorRuntimeHostClass();

      string appDomains = String.Empty;
      
      try
      {
        ArrayList list = new ArrayList();

        IntPtr enumHandle;
        host.EnumDomains(out enumHandle);
        
        while(true)
        {
          object domain;
          host.NextDomain(enumHandle, out domain);
          if(domain == null)
          {
            break;
          }

          if (appDomains != String.Empty)
          {
            appDomains += ", ";
          }

          appDomains += ((AppDomain)domain).FriendlyName;
        }
        
        host.CloseEnum(enumHandle);

      }
      finally
      {
        Marshal.ReleaseComObject(host);
      }

      Console.WriteLine("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
      Console.WriteLine("Active domains: " + appDomains);
      Console.WriteLine("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
    }
    
    #endregion

    #region protected methods
    /// <summary>
    /// Invokes method using reflection and un-wraps TargetInvocationException
    /// if it happens.
    /// </summary>
    /// <typeparam name="T">Type to cast result to.</typeparam>
    /// <param name="instance">Instance of object to execute method on.</param>
    /// <param name="method">Method to execute.</param>
    /// <param name="parameters">Parameters to pass to method.</param>
    /// <returns>Result of the method invocation.</returns>
    protected T Invoke<T>(object instance, MethodInfo method, params object[] parameters)
    {
      try
      {
        return (T)method.Invoke(instance, parameters);
      }
      catch (TargetInvocationException e)
      {
        Console.WriteLine("Exception occurred:\r\n{0}", e.InnerException.ToString());
        throw e.InnerException;
      }
    }

    /// <summary>
    /// Invokes static method using reflection and un-wraps TargetInvocationException
    /// if it happens.
    /// </summary>
    /// <typeparam name="T">Type to cast result to.</typeparam>    
    /// <param name="method">Method to execute.</param>
    /// <param name="parameters">Parameters to pass to method.</param>
    /// <returns>Result of the method invocation.</returns>
    protected T Invoke<T>(MethodInfo method, params object[] parameters)
    {
      try
      {
        return (T)method.Invoke(null, parameters);
      }
      catch (TargetInvocationException e)
      {
        Console.WriteLine("Exception occurred:\r\n{0}", e.InnerException.ToString());
        throw e.InnerException;
      }
    }

    /// <summary>
    /// Returns reference to the method to call.
    /// </summary>
    /// <typeparam name="T">Type to extract method from.</typeparam>
    /// <param name="name">Name of the method to extract.</param>
    /// <returns>Returns reference to the method.</returns>
    protected static MethodInfo GetMethod<T>(string name)
    {
      return GetMethod(typeof(T), name);
    }

    /// <summary>
    /// Returns reference to the method to call.
    /// </summary>
    /// <param name="type">Type to get method from.</param>
    /// <param name="name">Name of the method to extract.</param>
    /// <returns>Returns reference to the method.</returns>
    protected static MethodInfo GetMethod(Type type, string name)
    {
      MethodInfo result = type.GetMethod(name,
        BindingFlags.Public | BindingFlags.NonPublic |
        BindingFlags.Instance | BindingFlags.Static);

      Assert.IsNotNull(result, "Could find method {0} for type {1}.", name, type.Name);

      return result;
    }

    /// <summary>
    /// Returns reference to the field.
    /// </summary>
    /// <typeparam name="T">Type to extract field from.</typeparam>
    /// <param name="name">Name of the field to extract.</param>
    /// <returns>Returns reference to the field.</returns>
    protected static FieldInfo GetField<T>(string name)
    {
      return GetFieldRecursive(typeof(T), name);;
    }

    /// <summary>
    /// Recursevly searches for a fields in given type and all base types.
    /// </summary>
    /// <param name="type">Type to search for field in.</param>
    /// <param name="name">Field to search for.</param>
    /// <returns>Returns infromation about the field found or null.</returns>
    protected static FieldInfo GetFieldRecursive(Type type, string name)
    {
      FieldInfo result = type.GetField(name,
        BindingFlags.Public | BindingFlags.NonPublic |
        BindingFlags.Instance | BindingFlags.Static);

      if(result == null && type.BaseType != typeof(object))
      {
        return GetFieldRecursive(type.BaseType, name);
      }

      return result;
    }

    /// <summary>
    /// Reads value of the field from the object.
    /// </summary>
    /// <typeparam name="T">Type of the field.</typeparam>
    /// <param name="name">Name of the field to extract.</param>
    /// <param name="source">Object to read value from.</param>
    /// <returns>Returns value of the field.</returns>
    protected static T GetFieldValue<T>(string name, object source)
    {
      FieldInfo field = GetFieldRecursive(source.GetType(), name);

      Assert.IsNotNull(
        field,
        "Could not find field '{0}' in time '{1}'.",
        name,
        source.GetType());


      Assert.That(
        typeof(T).IsAssignableFrom(field.FieldType),
        "Field '{0}' has invalid type '{1}'.",
        name,
        field.FieldType);

      return (T)field.GetValue(source);
    }

    /// <summary>
    /// Reads value of the field from the object.
    /// </summary>
    /// <typeparam name="T">Type of the field.</typeparam>
    /// <param name="name">Name of the field to extract.</param>
    /// <param name="source">Object to read value from.</param>
    /// <returns>Returns value of the field.</returns>
    protected static T GetStaticFieldValue<T>(string name, Type type)
    {
      FieldInfo field = GetFieldRecursive(type, name);

      Assert.IsNotNull(
        field,
        "Could not find field '{0}' in time '{1}'.",
        name,
        type);

      Assert.That(
        typeof(T).IsAssignableFrom(field.FieldType),
        "Field '{0}' has invalid type '{1}'.",
        name,
        field.FieldType);

      return (T)field.GetValue(null);
    }

    /// <summary>
    /// Sets value of the field from the object.
    /// </summary>
    /// <typeparam name="T">Type of the field.</typeparam>
    /// <param name="name">Name of the field to extract.</param>
    /// <param name="source">Object to read value from.</param>
    /// <param name="value">New value to set.</param>
    protected static void SetFieldValue<T>(string name, object source, T value)
    {
      FieldInfo field = GetFieldRecursive(source.GetType(), name);

      Assert.IsNotNull(
        field,
        "Could not find field '{0}' in type '{1}'.",
        name,
        source.GetType());

      Assert.AreEqual(
        typeof(T),
        field.FieldType,
        "Field '{0}' has invalid type '{1}'.",
        name,
        field.FieldType);

      field.SetValue(source, value);
    }
    #endregion

    #region public methods
    /// <summary>
    /// This method is executed once before the testing starts. Base implementation enable logging.
    /// </summary>
    [TestFixtureSetUp]
    public virtual void SetUpFixture()
    {
      Console.WriteLine("Initializing test fixture '{0}'.", this.GetType().FullName);
      xmlHelper = new XmlHelper(MyAssembly);
      TraceSourceFactory.AddTraceListener(new ConsoleTraceListener());
      TraceSourceFactory.SwitchLevel = SourceLevels.Verbose;
    }

    /// <summary>
    /// This method is executed once after testing is completed (no matter successful or not).
    /// </summary>
    [TestFixtureTearDown]
    public virtual void TearDownFixture()
    {
      Console.WriteLine("Tearing down test fixture '{0}'.", this.GetType().FullName);
      ListAppDomains();
    }

    /// <summary>
    /// This method is executed before each particular test in the fixture.
    /// </summary>
    [SetUp]
    public virtual void SetUp()
    {
    }

    /// <summary>
    /// This method is after before each particular test in the fixture.
    /// </summary>
    [TearDown]
    public virtual void TearDown()
    {
    }
    #endregion
  }
}
