using System.Diagnostics;

namespace Utils.Log.Diagnostics
{
  /// <summary>
  /// Specific trace source for Mosaic logging
  /// </summary>
  public class CyclicTraceSource 
  {
    #region private fields

    /// <summary>
    /// Numeric identifier for all events
    /// </summary>
    const int messageId = 0;

   /// <summary>
   /// Instance of trace source that will manage all event records.
   /// </summary>
    private TraceSource traceSource;

    #endregion

    #region private methods ////////////////////////////////////////////////////

    /// <summary>
    /// This method trys to trace event, and if tracing fails, catchs the exception.
    /// </summary>
    /// <param name="eventType">Type of event for tracing.</param>
    /// <param name="messageId">Message ID.</param>
    /// <param name="message">Message that should be traced.</param>
    private void TryToTraceEvent(TraceEventType eventType, int messageId, string message)
    {
      try
      {        
        traceSource.TraceEvent(eventType, messageId, message);
      }
      catch
      {
        //Nothing to do here because if diagnostic fails, we cannot track this
        // and cannot allow diagnostic methods crach all application.
      }
    }

    #endregion

    #region public properties

    /// <summary>
    /// Gets the collection of trace listeners for the trace source.
    /// </summary>
    public TraceListenerCollection Listeners
    {
      get
      {
        return traceSource.Listeners;
      }
    }

    /// <summary>
    /// Gets or sets the source switch value.
    /// </summary>
    public SourceSwitch Switch
    {
      get
      {
        return traceSource.Switch;
      }
      set
      {
        traceSource.Switch = value;
      }
    }
    #endregion

    #region public methods

    /// <summary>
    /// CyclicTraceSource constructor.
    /// </summary>
    /// <param name="name">Trace source name.</param>
    public CyclicTraceSource(string name)
    {
      traceSource = new TraceSource(name);
    }

    /// <summary>
    /// Trace event with the Critical trace event type.
    /// </summary>
    /// <param name="strFormat">
    /// A string represents message text 
    /// (contains zero or more format items).
    /// </param>
    /// <param name="args">
    /// An array containing zero or more objects to format.
    /// </param>
    public void Critical(string strFormat, params object[] args)
    {
      string message = string.Format(strFormat, args);

      TryToTraceEvent(TraceEventType.Critical, messageId, message);
    }

    /// <summary>
    /// Trace event with the Error trace event type
    /// </summary>
    /// <param name="strFormat">
    /// A string representing message text 
    /// (contains zero or more format items).
    /// </param>
    /// <param name="args">
    /// An array containing zero or more objects to format.
    /// </param>
    public void Error(string strFormat, params object[] args)
    {
      string message = string.Format(strFormat, args);

      TryToTraceEvent(TraceEventType.Error, messageId, message);
    }    

    /// <summary>
    /// Trace event with the Warning trace event type.
    /// </summary>
    /// <param name="strFormat">
    /// A string representing message text 
    /// (contains zero or more format items).
    /// </param>
    /// <param name="args">
    /// An array containing zero or more objects to format.
    /// </param>
    public void Warning(string strFormat, params object[] args)
    {
      string message = string.Format(strFormat, args);

      TryToTraceEvent(TraceEventType.Warning, messageId, message);
    }

    /// <summary>
    /// Trace event with the Information trace event type.
    /// </summary>
    /// <param name="strFormat">
    /// A string representing message text 
    /// (contains zero or more format items).
    /// </param>
    /// <param name="args">
    /// An array containing zero or more objects to format.
    /// </param>
    public void Information(string strFormat, params object[] args)
    {
      string message = string.Format(strFormat, args);

      TryToTraceEvent(TraceEventType.Information, messageId, message);
    }

    /// <summary>
    /// Trace event with the Verbose trace event type.
    /// </summary>
    /// <param name="strFormat">
    /// A String represents message text 
    /// (contains zero or more format items).
    /// </param>
    /// <param name="args">
    /// An array containing zero or more objects to format.
    /// </param>
    public void Verbose(string strFormat, params object[] args)
    {
      string message = string.Format(strFormat, args);

      TryToTraceEvent(TraceEventType.Verbose, messageId, message);
    }

    #endregion
  } //end CyclicTraceSource
} //end namespace Utils.Diagnostics