using System;

namespace HarryBonesSolutions.Performance
{
	/// <summary>
	/// Holds a reference to a delegate targeting a method
	/// to be executed, as well as set-up and tear-down methods.
	/// </summary>
	/// <remarks>
	/// The only required value in this class is Method. If SetUp 
	/// or TearDown are null, they will simply be skipped. The same 
	/// holds true for the arguments to all three delegates.
	/// 
	/// If Iterations is not set here, it must be specified when an
	/// instance of this class is passed to ExecutionTimer.RunTimedMethod.
	/// </remarks>
	public class TimedMethod
	{
		private Delegate	  _method;
		private Delegate	  _setUp;
		private Delegate	  _tearDown;
		private object[]	  _args;
		private object[]	  _setUpArgs;
		private object[]	  _tearDownArgs;
		private int			  _iterations;
		private string		  _displayName;

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class.
		/// </summary>
		public TimedMethod()
		{
			_method = null;
			_setUp = null;
			_tearDown = null;
			_args = null;
			_setUpArgs = null;
			_tearDownArgs = null;
			_iterations = 0;
			_displayName = string.Empty;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		public TimedMethod(Delegate method) : this()
		{
			_method = method;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method and iterations.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(Delegate method, int iterations) : this(method)
		{
			_iterations = iterations;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method and arguments.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args"></param>
		public TimedMethod(Delegate method, object[] args) : this(method)
		{
			_args = args;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method, arguments, and iterations.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(Delegate method, object[] args, int iterations) : this(method, iterations)
		{
			_args = args;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified methodm set-up, and tear-down.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="setUp"></param>
		/// <param name="tearDown"></param>
		public TimedMethod(Delegate method, Delegate setUp, Delegate tearDown) : this(method)
		{
			_setUp = setUp;
			_tearDown = tearDown;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method, set-up, teard-down, and iterations.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="setUp"></param>
		/// <param name="tearDown"></param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(Delegate method, Delegate setUp, Delegate tearDown, int iterations) : this(method, setUp, tearDown)
		{
			_iterations = iterations;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method, arguments, set-up, set-up arguments, tear-down,
		/// and tear-down arguments.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="setUpArgs">The arguemtns to be supplied to the set-up method.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		/// <param name="tearDownArgs">The arguemtns to be supplied to the tear-down method.</param>
		public TimedMethod(Delegate method, object[] args, Delegate setUp, 
			object[] setUpArgs, Delegate tearDown, object[] tearDownArgs) : this(method, setUp, tearDown)
		{
			_args = args;
			_setUpArgs = setUpArgs;
			_tearDownArgs = tearDownArgs;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified method, arguments, set-up, set-up arguments, tear-down,
		/// tear-down arguments, and iterations.
		/// </summary>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="setUpArgs">The arguemtns to be supplied to the set-up method.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		/// <param name="tearDownArgs">The arguemtns to be supplied to the tear-down method.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(Delegate method, object[] args, Delegate setUp, 
			object[] setUpArgs, Delegate tearDown, object[] tearDownArgs, int iterations) 
			: this(method, args, setUp, setUpArgs, tearDown, tearDownArgs)
		{
			_iterations = iterations;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name and method.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		public TimedMethod(string displayName, Delegate method) : this(method)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, and iterations.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(string displayName, Delegate method, int iterations) : this(method, iterations)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, arguments.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		public TimedMethod(string displayName, Delegate method, object[] args) : this(method, args)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, arguments, and iterations.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(string displayName, Delegate method, object[] args, int iterations) : this(method, args, iterations)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, set-up, and tear-down.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		public TimedMethod(string displayName, Delegate method, Delegate setUp, Delegate tearDown) : this(method, setUp, tearDown)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, set-up, tear-down, and iterations.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(string displayName, Delegate method, Delegate setUp, Delegate tearDown, int iterations) : this(method, setUp, tearDown, iterations)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, arguments, set-up, set-up arguments, tear-down,
		/// and tear-down arguments.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="setUpArgs">The arguemtns to be supplied to the set-up method.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		/// <param name="tearDownArgs">The arguemtns to be supplied to the tear-down method.</param>
		public TimedMethod(string displayName, Delegate method, object[] args, Delegate setUp, 
			object[] setUpArgs, Delegate tearDown, object[] tearDownArgs) : this(method, args, setUp, setUpArgs, tearDown, tearDownArgs)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="TimedMethod"/> class with the 
		/// specified display name, method, arguments, set-up, set-up arguments, tear-down,
		/// tear-down arguments, and iterations.
		/// </summary>
		/// <param name="displayName">The name to be used when displaying execution results.</param>
		/// <param name="method">The main method to be timed.</param>
		/// <param name="args">The arguemtns to be supplied to the timed method.</param>
		/// <param name="setUp">The method to be run before the method being timed.</param>
		/// <param name="setUpArgs">The arguemtns to be supplied to the set-up method.</param>
		/// <param name="tearDown">The method to be run after the method being timed.</param>
		/// <param name="tearDownArgs">The arguemtns to be supplied to the tear-down method.</param>
		/// <param name="iterations">The number of iterations to run the method.</param>
		public TimedMethod(string displayName, Delegate method, object[] args, Delegate setUp, 
			object[] setUpArgs, Delegate tearDown, object[] tearDownArgs, int iterations) 
			: this(method, args, setUp, setUpArgs, tearDown, tearDownArgs, iterations)
		{
			_displayName = displayName;
		}

		/// <summary>
		/// Gets or sets the arguments to be passed to the timed delegate
		/// </summary>
		public object[] Args
		{
			get{ return _args; }
			set{ _args = value; }
		}

		/// <summary>
		/// Gets or sets the arguments to be passed to the set-up delegate
		/// </summary>
		public object[] SetUpArgs
		{
			get{ return _setUpArgs; }
			set{ _setUpArgs = value; }
		}

		/// <summary>
		/// Gets or sets the arguments to be passed to the tear-down delegate
		/// </summary>
		public object[] TearDownArgs
		{
			get{ return _tearDownArgs; }
			set{ _tearDownArgs = value; }
		}

		/// <summary>
		/// Gets or sets the delegate targeting the main method to be timed when RunTimedMethod is called.
		/// </summary>
		/// <remarks>If this delegate targets more then one method, only the first will be called.</remarks>
		public Delegate Method
		{
			get{ return _method; }
			set{ _method = value; }
		}

		/// <summary>
		/// Gets or sets the delegate targeting the method run before each iteration of the timed method
		/// when RunTimedMethod is called.  Execution time for this delegate is not included in the
		/// run.  
		/// </summary>
		/// <remarks>If this delegate targets more then one method, only the first will be called.</remarks>
		public Delegate SetUp
		{
			get{ return _setUp; }
			set{ _setUp = value; }
		}

		/// <summary>
		/// Gets or sets the delegate targeting the method run after each iteration of the timed method
		/// when RunTimedMethod is called.  Execution time for this delegate is not included in the
		/// run.
		/// </summary>
		/// <remarks>If this delegate targets more then one method, only the first will be called.</remarks>
		public Delegate TearDown
		{
			get{ return _tearDown; }
			set{ _tearDown = value; }
		}

		/// <summary>
		/// Gets or sets the number of times the method targeted by
		/// the Method property should be executed.
		/// </summary>
		public int Iterations
		{
			get{ return _iterations; }
			set{ _iterations = value; }
		}

		/// <summary>
		/// Gets or sets the name to be displayed when the 
		/// method execution statistics are displayed.
		/// </summary>
		public string DisplayName
		{
			get{ return _displayName; }
			set{ _displayName = value;	}
		}
	}
}