using System;
using System.Runtime.Serialization;

namespace HarryBonesSolutions.Performance
{
	/// <summary>
	/// The exception thrown when a series could not be run for a specific <see cref="TimedMethod"/> object.
	/// </summary>
	[Serializable]
	public sealed class SeriesAbortedException : ApplicationException, ISerializable
	{
		private string _seriesName = string.Empty;

		/// <summary>
		/// Initializes a new instance of the <see cref="SeriesAbortedException"/> class.
		/// </summary>
		public SeriesAbortedException() : base(){}

		/// <summary>
		/// Initializes a new instance of the <see cref="SeriesAbortedException"/> class
		///  with the specified message.
		/// </summary>
		/// <param name="message">The message explaining why the series was aborted.</param>
		public SeriesAbortedException(string message) : base(message){}

		/// <summary>
		/// Initializes a new instance of the <see cref="SeriesAbortedException"/> class
		///  with the specified message and inner exception.
		/// </summary>
		/// <param name="message">The message explaining why the series was aborted.</param>
		/// <param name="innerException">The exception that caused this exception.</param>
		public SeriesAbortedException(string message, Exception innerException) : base (message, innerException){}

		/// <summary>
		/// Initializes a new instance of the <see cref="SeriesAbortedException"/> class
		///  with the specified message and series name.
		/// </summary>
		/// <param name="message">The message explaining why the series was aborted.</param>
		/// <param name="seriesName">The name of the series being aborted.</param>
		public SeriesAbortedException(string message, string seriesName) : this (message, seriesName, null){}

		/// <summary>
		/// Initializes a new instance of the <see cref="SeriesAbortedException"/> class
		///  with the specified message, series name, and inner exception.
		/// </summary>
		/// <param name="message">The message explaining why the series was aborted.</param>
		/// <param name="seriesName">The name of the series being aborted.</param>
		/// <param name="innerException">The exception that caused this exception.</param>
		public SeriesAbortedException(string message, string seriesName, Exception innerException) : base (message, innerException)
		{
			_seriesName = seriesName;
		}


		// Constructor needed for deserialization.
		private SeriesAbortedException(SerializationInfo info, StreamingContext context) : base(info, context)
		{
			_seriesName = info.GetString("SeriesName");
		}

		/// <summary>
		/// Gets the name of the series being aborted.
		/// </summary>
		public string SeriesName
		{
			get{ return _seriesName; }
		}

		#region ISerializable Members

		/// <summary>
		/// Implements the <see cref="ISerializable"/> interface and sets the 
		/// data needed to serialize the <see cref="NameExistsException"/> object.
		/// </summary>
		/// <param name="info">A <see cref="SerializationInfo"/> object that contains the 
		/// information required to serialize the <see cref="SeriesAbortedException"/> instance. 
		/// </param>
		/// <param name="context">A <see cref="StreamingContext"/> object that 
		/// contains the source and destination of the serialized stream 
		/// associated with the <see cref="SeriesAbortedException"/> instance.</param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			info.AddValue("SeriesName", _seriesName, typeof(string));

			base.GetObjectData(info, context);
		}

		#endregion
	}
}