using System;
using System.Windows.Forms;
using System.Threading;
using System.IO;

namespace Uznavayko.TrayTool
{
  /// <summary>
  /// Main class with program entry point.
  /// </summary>
  internal class TrayTool
  {
    #region private fields

    /// <summary>
    /// Status icon shown in tray.
    /// Every state of Mosaic Host instance is represented by
    /// dedicated tray icon.
    /// </summary>
    private static StatusTrayIcon statusIcon;

    /// <summary>
    /// The event to be raised when someone wants TrayTool to be closed.
    /// E.g. uninstallation may want to do so.
    /// </summary>
    private static EventWaitHandle quitRequested;

    /// <summary>
    /// Timer which handles request to quit.
    /// </summary>
    private static System.Threading.Timer quitTimer;

    #endregion

    #region private methods

    /// <summary>
    /// Quits application freeing all resources.
    /// </summary>
    private static void Quit()
    {
      statusIcon.Close();
      quitRequested.Close();
      Environment.Exit(0);
    }

    /// <summary>
    /// Handles quit timer.
    /// </summary>
    /// <param name="state">Ignored.</param>
    private static void OnQuitTimer(object state)
    {
      if (quitRequested.WaitOne(0, false))
      {
        Quit();
      }
    }

    /// <summary>
    /// Returns name for event used to close application request.
    /// </summary>
    private static string GetApplicationEventName()
    {
      // Removes slashes and semicolon as otherwise event throws
      // DirectoryNotFoundException for unknown reasons.
      return Application.ExecutablePath.Replace(":", "").Replace("\\", "");
    }

    /// <summary>
    /// The main entry point for the application.
    /// </summary>
    [STAThread]
    private static void Main(string[] args)
    {
      if (args.Length == 1)
      {
        if (args[0].Equals("exit", StringComparison.InvariantCultureIgnoreCase))
        {
          try
          {
            quitRequested = EventWaitHandle.OpenExisting(GetApplicationEventName());
            quitRequested.Set();
          }
          // Do nothing with exceptions which may occur.
          // In this case we just cannot close the app.
          catch (WaitHandleCannotBeOpenedException)
          {
          }
          catch (IOException)
          {
          }
          catch (UnauthorizedAccessException)
          {
          }
          return;
        }
      }
      else
      {
        quitRequested = new EventWaitHandle(
          false,
          EventResetMode.ManualReset,
          GetApplicationEventName());
        quitTimer = new System.Threading.Timer(new TimerCallback(OnQuitTimer), null, 0, 1000);
      }


      Application.EnableVisualStyles();
      Application.SetCompatibleTextRenderingDefault(false);

      statusIcon = new StatusTrayIcon();
      statusIcon.ExitRequested += delegate { Quit(); };
      HostWatcher.HostStatusChanged += statusIcon.OnInstanceStatus;

      // No form should be shown on startup
      Application.Run();
    }

    #endregion
  } // end HostMaintenanceTrayTool
} // end namespace Uznavayko.TrayTool