using System;
using System.Windows.Forms;
using Utils.Service;

namespace Uznavayko.TrayTool
{
  /// <summary>
  /// This class represents icon in tray which shows current Host status,
  /// Host exception details and handles application exit.
  /// </summary>
  internal sealed class StatusTrayIcon
  {
    #region nested classes /////////////////////////////////////////////////////
    /// <summary>
    /// Arguments passed by event notifying about Host changes.
    /// </summary>
    public class HostStatusEventArgs : EventArgs
    {
      /// <summary>
      /// Current Host status.
      /// </summary>
      public readonly Status Status;

      /// <summary>
      /// Friendly message about Host status which can be shown to user.
      /// </summary>
      public readonly string FriendlyMessage = string.Empty;

      /// <summary>
      /// Not user-friendly message with Host exception (if any).
      /// Shown by clicking on tray icon with left key (or on choosing from menu).
      /// </summary>
      public readonly string ExceptionMessage = string.Empty;


      /// <summary>
      /// Constructor from status only.
      /// </summary>
      public HostStatusEventArgs(Status status)
      {
        Status = status;
      }

      /// <summary>
      /// Constructor from status and friendly message.
      /// </summary>
      public HostStatusEventArgs(Status status, string friendlyMessage)
      {
        Status = status;
        FriendlyMessage = friendlyMessage;
      }

      /// <summary>
      /// Constructor from both status, friendly message and exception message.
      /// </summary>
      public HostStatusEventArgs(Status status, string friendlyMessage, string exceptionMessage)
      {
        Status = status;
        FriendlyMessage = friendlyMessage;
        ExceptionMessage = exceptionMessage;
      }
    }
    #endregion

    #region private fields /////////////////////////////////////////////////////

    /// <summary>
    /// System tray icon representing selected Mosaic Host instance status
    /// </summary>
    private NotifyIcon trayStatusIcon = new NotifyIcon();

    /// <summary>
    /// Last known Host status.
    /// </summary>
    private HostStatusEventArgs lastStatus;

    #endregion

    #region private methods /////////////////////////////////////////////////////

    /// <summary>
    /// Shows details on Host status in message box.
    /// </summary>
    private void ShowHostStatus(object sender, EventArgs args)
    {
      string message;

      if (lastStatus == null)
      {
        message = "The Uznavayko host is not connected";
      }
      else
      {
        message =
            string.IsNullOrEmpty(lastStatus.ExceptionMessage) ?
            lastStatus.FriendlyMessage :
            lastStatus.ExceptionMessage;
      }

      MessageBox.Show(message, "Details on current Uznavayko host status",
        MessageBoxButtons.OK, MessageBoxIcon.Information);
    }

    /// <summary>
    /// Handles click on tray icon showing last known Host status.
    /// </summary>
    private void TrayIconMouseClick(object sender, MouseEventArgs e)
    {
      if (e.Button == MouseButtons.Left)
      {
        ShowHostStatus(null, null);
      }
    }

    /// <summary>
    /// Handles clicking on Exit tray menu item. Exits the application.
    /// </summary>
    private void OnExit(object sender, EventArgs e)
    {
      if (ExitRequested != null)
      {
        ExitRequested(sender, e);
      }
    }

    #endregion

    #region public methods

    /// <summary>
    /// Event which is fired when a user have chosen Exit tray menu item.
    /// </summary>
    public event EventHandler ExitRequested;

    /// <summary>
    /// Removes tray icon from the tray.
    /// </summary>
    public void Close()
    {
      trayStatusIcon.Dispose();
    }

    /// <summary>
    /// Handler of an event of Host status change.
    /// </summary>
    /// <param name="sender">Ignored.</param>
    /// <param name="e">Host status event args handling status, message and exception.</param>
    public void OnInstanceStatus(object sender, EventArgs e)
    {
      lastStatus = (HostStatusEventArgs)e;
      trayStatusIcon.BalloonTipText = lastStatus.FriendlyMessage;

      switch (lastStatus.Status)
      {
        case Status.Error:
          {
            trayStatusIcon.Icon =
              StatusIcons.TrayIconStatusErrors;
            trayStatusIcon.ShowBalloonTip(2, "Error", 
              "Current user did not pass the validation process.",
              ToolTipIcon.Error);
            break;
          }
        case Status.Ok:
          {
            trayStatusIcon.Icon =
              StatusIcons.TrayIconStatusRunning;
            trayStatusIcon.ShowBalloonTip(1);
            break;
          }
        case Status.Warning:
          {
            trayStatusIcon.Icon =
              StatusIcons.TrayIconStatusWarning;
            trayStatusIcon.ShowBalloonTip(1);
            break;
          }
      }
    }

    public bool notTheFirstTime;
    /// <summary>
    /// Constructor initializing icon in tray.
    /// </summary>
    public StatusTrayIcon()
    {
      trayStatusIcon.Icon = StatusIcons.TrayIconStatusErrors;
      trayStatusIcon.Visible = true;

      ContextMenu menu = new ContextMenu(
        (
          new MenuItem[]
          {
            new MenuItem("See status details", ShowHostStatus),
            new MenuItem("-"),
            new MenuItem("E&xit", OnExit)
          }
        ));

      trayStatusIcon.ContextMenu = menu;
      trayStatusIcon.BalloonTipText = "Retrieving status...";
      trayStatusIcon.MouseClick += TrayIconMouseClick;
    }

    #endregion
  } // end StatusTrayIcon
} // end namespace Uznavayko.TrayTool