using System;
using System.Timers;
using System.ServiceModel;
using Utils.Service;

namespace Uznavayko.TrayTool
{
  /// <summary>
  /// This class watches Host status via WCF connection from configuration file.
  /// It fires event in case the status was changed.
  /// </summary>
  internal class HostWatcher
  {
    #region private fields

    /// <summary>
    /// A value for periodic update of status in milliseconds.
    /// </summary>
    private const int watcherTimerInterval = 1000;

    /// <summary>
    /// Timer which periodically polls the request for update status.
    /// </summary>
    private static Timer watcherTimer = new Timer(watcherTimerInterval);

    #endregion
    #region private methods /////////////////////////////////////////////////////

    /// <summary>
    /// Asks remote Host for its status via WCF.
    /// Endpoint used is default one from configuration file.
    /// </summary>
    private static void UpdateHostStatus()
    {
      Status status = Status.Warning;
      string friendlyMessage = string.Empty;
      string hostException = string.Empty;
      ServiceProxy hostProxy = new ServiceProxy();

      try
      {
        status = hostProxy.GetStatus();
      }
      catch (EndpointNotFoundException)
      {
        status = Status.Error;
        hostException = "The host is not started";
      }
      catch (ArgumentException)
      {
        hostException = "The host is failed";
      }
      catch (CommunicationException)
      {
        // service does not respond, show this as 'not started'.
        hostException = "The host is failed";
      }

      finally
      {
        if (hostProxy.State != CommunicationState.Faulted)
        {
          ((IDisposable)hostProxy).Dispose();
        }
      }

      friendlyMessage = Enum.GetName(typeof(Status),status);

      FireHostStatusChanged(status, friendlyMessage, hostException);
    }

    /// <summary>
    /// Fires the host status changed event.
    /// </summary>
    /// <param name="status">if set to <c>true</c> [status].</param>
    /// <param name="friendlyMessage">The friendly message.</param>
    /// <param name="hostException">The host exception.</param>
    private static void FireHostStatusChanged(
      Status status, string friendlyMessage, 
      string hostException)
    {
      if (HostStatusChanged != null)
      {
        HostStatusChanged(null,
          new StatusTrayIcon.HostStatusEventArgs(status,
            friendlyMessage,
            hostException));
      }
    }

    /// <summary>
    /// Handles 'host watcher' timer ticks.
    /// </summary>
    private static void OnWatcherTimerElapsed(
      object sender,
      ElapsedEventArgs e)
    {
      UpdateHostStatus();
    }

    #endregion

    #region public methods

    /// <summary>
    /// An event to fire when status is changed.
    /// </summary>
    public static event EventHandler<StatusTrayIcon.HostStatusEventArgs> HostStatusChanged;

    /// <summary>
    /// Static constructor (we only need single instance of watcher).
    /// Starts watcher timer.
    /// </summary>
    static HostWatcher()
    {
      UpdateHostStatus();
      watcherTimer.Elapsed += OnWatcherTimerElapsed;
      watcherTimer.Start();
    }
    #endregion
  } // end HostWatcher
} // end namespace Uznavayko.TrayTool