using System;
using System.Collections.Generic;
using System.Text;

namespace MouseAnalyzer.MouseTraits
{

    class MovementMouseTrait : ITrait
    {
        private List<MovementTrajectoryPoint> _trajectoryPoints;
        private CubicSpline _splineApproximation;

        public int TrajectoryPointsCount
        {
            get
            {
                return _trajectoryPoints.Count;
            }
        }

        public MovementMouseTrait()
        {
            _trajectoryPoints = new List<MovementTrajectoryPoint>();
            _splineApproximation = null;
        }

        public void AddPoint(int time, int x, int y)
        {
            _trajectoryPoints.Add(new MovementTrajectoryPoint(time, x, y));
        }

        public void Clear()
        {
            _trajectoryPoints.Clear();
        }

        public double GetSpeedAverage()
        {
            if (_trajectoryPoints.Count > 0)
            {
                int startTime = _trajectoryPoints[0].Time;
                int finishTime = _trajectoryPoints[_trajectoryPoints.Count - 1].Time;
                if (finishTime != startTime)
                {
                    double totalTime = Math.Abs(finishTime - startTime);
                    return GetTrajectoryLength() / totalTime;
                }
            }
            return 0;
        }

        public double GetAccelerationMax()
        {
            if (_trajectoryPoints.Count > 0)
            {
                double accelerationMax = 0;
                int pointIndex = 0;
                double prevSpeed = 0;
                while (pointIndex < _trajectoryPoints.Count - 1)
                {
                    int currTime = _trajectoryPoints[pointIndex].Time;
                    int nextPointIndex = pointIndex + 1;
                    double pointsDistance = 0;
                    while ((nextPointIndex < _trajectoryPoints.Count) && (currTime == _trajectoryPoints[nextPointIndex].Time))
                    {
                        pointsDistance += GetTwoPointsDistance(_trajectoryPoints[nextPointIndex - 1], _trajectoryPoints[nextPointIndex]);
                        nextPointIndex++;
                    }
                    if (nextPointIndex < _trajectoryPoints.Count)
                    {
                        double currSpeed = pointsDistance / Math.Abs(_trajectoryPoints[nextPointIndex].Time - currTime);
                        if (currSpeed - prevSpeed > accelerationMax)
                        {
                            accelerationMax = currSpeed - prevSpeed;
                        }
                        prevSpeed = currSpeed;
                    }
                    pointIndex++;
                }
                return accelerationMax;
            }
            return 0;
        }

        public double GetCurvatureAverage()
        {
            if (_trajectoryPoints.Count == 0)
            {
                return 0;
            }
            //      (C2)  
            if (_splineApproximation == null)
            {
                UpdateCubicSplineApproximation();
            }
            double curvatureAverage = 0;
            for (int i = 0; i < _splineApproximation.NodesCount; i++)
            {
                curvatureAverage += _splineApproximation.CalculateCurvature(i);
            }
            return curvatureAverage / _splineApproximation.NodesCount;
        }

        public double GetCurvatureMax()
        {
            if (_trajectoryPoints.Count == 0)
            {
                return 0;
            }
            //      (C2) 
            if (_splineApproximation == null)
            {
                UpdateCubicSplineApproximation();
            }
            double curvatureMax = 0;
            for (int i = 0; i < _splineApproximation.NodesCount; i++)
            {
                double currentCurvature = _splineApproximation.CalculateCurvature(i);
                if (currentCurvature > curvatureMax)
                {
                    curvatureMax = currentCurvature;
                }
            }
            return curvatureMax;
        }

        public double GetFinalPointAccuracy()
        {
            if (_trajectoryPoints.Count > 1)
            {
                int lastPointIndex = _trajectoryPoints.Count - 1;
                int currentPointIndex = lastPointIndex - 1;
                double prevLength = GetTwoPointsDistance(_trajectoryPoints[lastPointIndex], _trajectoryPoints[currentPointIndex]);
                currentPointIndex--;
                double currentLength = GetTwoPointsDistance(_trajectoryPoints[lastPointIndex], _trajectoryPoints[currentPointIndex]);
                while ((prevLength < currentLength) && (currentPointIndex > 0))
                {
                    currentPointIndex--;
                    prevLength = currentLength;
                    currentLength = GetTwoPointsDistance(_trajectoryPoints[lastPointIndex], _trajectoryPoints[currentPointIndex]);
                }
                if (prevLength > currentLength)
                {
                    return prevLength / GetTrajectoryLength();
                }
            }
            return 0;
        }

        public double GetTrajectoryLength()
        {
            double totalLength = 0;
            if (_trajectoryPoints.Count > 0)
            {
                for (int pointIndex = 0; pointIndex < _trajectoryPoints.Count - 1; pointIndex++)
                {
                    totalLength += GetTwoPointsDistance(_trajectoryPoints[pointIndex], _trajectoryPoints[pointIndex + 1]);
                }
            }
            return totalLength;
        }

        public double GetBeginToEndDistance()
        {
            if (_trajectoryPoints.Count > 0)
            {
                return GetTwoPointsDistance(_trajectoryPoints[0], _trajectoryPoints[_trajectoryPoints.Count - 1]);
            }
            return 0;
        }

        private double GetTwoPointsDistance(MovementTrajectoryPoint point1, MovementTrajectoryPoint point2)
        {
            double dx = point1.X - point2.X;
            double dy = point1.Y - point2.Y;
            return Math.Sqrt(dx * dx + dy * dy);
        }

        private void UpdateCubicSplineApproximation()
        {
            double[] x = new double[_trajectoryPoints.Count];
            double[] y = new double[_trajectoryPoints.Count];
            for (int i = 0; i < _trajectoryPoints.Count; i++)
            {
                x[i] = _trajectoryPoints[i].X;
                y[i] = _trajectoryPoints[i].Y;
            }
            _splineApproximation = CubicSpline.InterpolatePolyline(x, y);
        }

        #region ITrait Members

        public double[] GenerateTraitVector()
        {
            return new double[] 
            {
                this.GetAccelerationMax(),
                this.GetCurvatureAverage(),
                this.GetCurvatureMax(),
                this.GetFinalPointAccuracy(),
                this.GetSpeedAverage(),
                this.GetTrajectoryLength()
            };
        }

        #endregion
    }
}
