﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml.Serialization;

namespace XmlSerialization
{
  internal enum SerializeMode
  {
    Notes,
    Settings
  }

  /// <summary>
  /// This class is used to write and read XML doc with notes
  /// </summary>
  public sealed class XmlSerialization : IDisposable
  {
    #region Props

    /// <summary>
    /// The name of XML file.
    /// </summary>
    private string xmlFileName;

    /// <summary>
    /// The Stream to read XML with.
    /// </summary>
    private static Stream stream;

    /// <summary>
    ///  Serializes and deserializes objects into and from XML documents. 
    ///	 The System.Xml.Serialization.XmlSerializer enables you to control 
    ///	 how objects are encoded into XML.
    /// </summary>
    private XmlSerializer serializer;

    /// <summary>
    ///  Represents a writer that can write a sequential series of characters. 
    /// We use it to write XML with.
    /// </summary>
    private TextWriter writer;

    private UznavaykoPasFile noteList;

    /// <summary>
    /// Gets the note list that contains all notes.
    /// </summary>
    /// <value>The note list.</value>
    public UznavaykoPasFile NoteList
    {
      get { return noteList; }
    }

    #endregion

    /// <summary>
    /// Initializes a new instance of the <see cref="XmlSerialization"/> class.
    /// Reading xml if there is no file, create it.
    /// </summary>
    public XmlSerialization(string xmlFileName)
    {
      this.xmlFileName = xmlFileName;

      Init();
    }

    /// <summary>
    /// Inits this instance.
    /// </summary>
    private void Init()
    {
      serializer = new XmlSerializer(typeof(UznavaykoPasFile));

      if (serializer == null)
      {
        throw new ArgumentException();
      }

      try
      {
        noteList = (UznavaykoPasFile)serializer.
          Deserialize(GetXmlFileStream(xmlFileName));
      }
      catch (InvalidOperationException)
      {
        noteList = new UznavaykoPasFile();
      }
    }

    /// <summary>
    /// Adds the note to collection, but not in xml file.
    /// </summary>
    /// <param name="note">The note.</param>
    internal void AddNote(Word word)
    {
      noteList.AddItem(word);
    }

    /// <summary>
    /// Deletes the note from collectiom, but not in xml file.
    /// </summary>
    /// <param name="note">The note.</param>
    internal void DeleteNote(Word word)
    {
      noteList.DeleteItem(word);
    }

    /// <summary>
    /// Writes all notes to XML.
    /// </summary>
    public void WriteToFile()
    {
      stream.Close();
      try
      {
        writer = new StreamWriter(xmlFileName);
      }
      catch (IOException)
      {
        return;
      }

      serializer.Serialize(writer, noteList);
      writer.Close();

    }

    /// <summary>
    /// Gets the XML stream from file.
    /// </summary>
    /// <param name="name">The name of file.</param>
    /// <returns></returns>
    private static Stream GetXmlFileStream(string name)
    {
      stream = new FileStream(name, FileMode.OpenOrCreate, FileAccess.Read);
      return stream;
    }

    #region IDisposable Members

    public void Dispose()
    {
      if (stream != null)
      {
        stream.Dispose();
      }
      if (writer != null)
      {
        writer.Dispose();
      }
      //we do not dispose noteList because we need it))
      //if (noteList != null)
      //{
      //  foreach (Word item in noteList.Words)
      //  {
      //    item.Dispose();
      //  }
      //}
    }

    #endregion

  } //end XmlSerialization
}
