using System;
using System.Text;
using System.Xml;
using System.IO;
using NUnit.Framework;
using System.Reflection;

namespace Utils.Diagnostics
{
  /// <summary>
  /// Assists in XmlReader creation
  /// </summary>
  public class XmlHelper
  {
    #region private fields
    /// <summary>
    /// Private field used to access assembly information.
    /// </summary>
    private readonly Assembly assembly;

    /// <summary>
    /// Resource prefix to use.
    /// </summary>
    private readonly string resourcePrefix;
    #endregion

    #region public methods
    
    #region initialization
    /// <summary>
    /// Creates an instance of XmlHelper.
    /// </summary>
    /// <param name="resourceAssembly">
    /// Assembly which contains resource files to read.
    /// </param>
    internal XmlHelper(Assembly resourceAssembly)
    {
      if (resourceAssembly == null)
      {
        throw new ArgumentNullException("resourceAssembly");
      }
      assembly = resourceAssembly;
      resourcePrefix = assembly.GetName().Name;
    } 
    #endregion

    #region loading
    /// <summary>
    /// Creates XML reader from the string content.
    /// </summary>
    public XmlReader ReaderFromString(string content)
    {
      XmlReaderSettings setting = new XmlReaderSettings();
      setting.CloseInput = true;

      XmlReader result = XmlReader.Create(new StringReader(content), setting);
      result.Read();

      return result;
    }

    /// <summary>
    /// Creates an XmlReader from the assembly resource.
    /// </summary>
    /// <param name="resourceName">Resource name to read resource from.</param>
    /// <returns>Returns XmlReader based on the resource stream.</returns>
    public XmlReader ReaderFromResources(string resourceName)
    {
      Stream resource = assembly.GetManifestResourceStream(resourcePrefix + '.' + resourceName);
      if (resource == null)
      {
        Assert.Fail("Failed to extract resource stream for '{0}'", resourceName);
      }
      return XmlReader.Create(resource);
    }

    /// <summary>
    /// Creates an XmlDocument from the assembly resource.
    /// </summary>
    /// <param name="resourceName">Resource name to read resource from.</param>
    /// <returns>Returns XmlDocument based on the resource stream.</returns>
    public XmlDocument DocumentFromResources(string resourceName)
    {
      XmlDocument result = new XmlDocument();
      result.Load(ReaderFromResources(resourceName));
      return result;
    } 
    #endregion

    #region comparasion
    /// <summary>
    /// Compares two XML files ignoring minor differences
    /// </summary>
    /// <param name="file1">Name of the first file to compare.</param>
    /// <param name="file2">Name of the second file to compare.</param>
    /// <returns>Returns true if files contains equal XML.</returns>
    public void CompareXmlFiles(string file1, string file2)
    {
      // Load two XmlDocuments from file
      XmlDocument document1 = new XmlDocument(), document2 = new XmlDocument();
      document1.Load(Environment.CurrentDirectory + '\\' + file1);
      document2.Load(Environment.CurrentDirectory + '\\' + file2);

      CompareXmlDocuments(document1, document2);
    }

    /// <summary>
    /// Compares to XmlDocument objects for being equivalent.
    /// </summary>
    public void CompareXmlDocuments(XmlDocument document1, XmlDocument document2)
    {
      // Writes two documents to string with identical XmlWriters
      StringBuilder builder1 = new StringBuilder(), builder2 = new StringBuilder();
      XmlWriterSettings settings = new XmlWriterSettings();
      settings.Indent = true;
      XmlWriter writer1 = XmlWriter.Create(builder1, settings), writer2 = XmlWriter.Create(builder2, settings);
      document1.WriteTo(writer1);
      document2.WriteTo(writer2);
      writer1.Flush();
      writer2.Flush();
      writer1.Close();
      writer2.Close();

      // Compare resulting strings
      Assert.AreEqual(builder1.ToString(), builder2.ToString());
    }

    /// <summary>
    /// Compares two XmlReaders ignoring minor differences
    /// </summary>    
    public void CompareXmlReaders(XmlReader reader1, XmlReader reader2)
    {
      // Load two XmlDocuments from file
      XmlDocument document1 = new XmlDocument(), document2 = new XmlDocument();
      document1.Load(reader1);
      document2.Load(reader2);

      CompareXmlDocuments(document1, document2);
    }

    /// <summary>
    /// Compares XmlReader with XML provided as string.
    /// </summary>    
    public void CompareXmlReaderWithString(XmlReader reader1, string content)
    {
      // Load two XmlDocuments from file
      XmlDocument document1 = new XmlDocument(), document2 = new XmlDocument();
      document1.Load(reader1);
      document2.Load(ReaderFromString(content));

      CompareXmlDocuments(document1, document2);
    }

    /// <summary>
    /// Compares XmlReader with XML from the resource stream.
    /// </summary>    
    public void CompareXmlReaderWithResource(XmlReader reader1, string resourceName)
    {
      // Load two XmlDocuments from file
      XmlDocument document1 = new XmlDocument(), document2 = new XmlDocument();
      document1.Load(reader1);
      document2.Load(ReaderFromResources(resourceName));

      CompareXmlDocuments(document1, document2);
    }
    #endregion

    #endregion
  }
}
