﻿using System;
using System.IO;
using NeuronDotNet.Core;
using System.Collections.Generic;
using NeuronDotNet.Core.Initializers;
using NeuronDotNet.Core.Backpropagation;
using System.Runtime.Serialization.Formatters.Binary;
using NeuronDotNet.Core.SOM;
using Utils.Diagnostics;
using Utils.Log.Diagnostics;

namespace Uznavayko.NeyronNetwork.UznavaykoNetwork
{
  /// <summary>
  /// This is the class for neyron network for the Uznavayko project.
  /// 
  /// Uznavaykos neyron network inputs are:
  /// 1)Daytime
  /// 2)Day of the week
  /// 3)First char
  /// 4)Second char
  /// 5)Time between first and second char
  /// 
  /// we can give varioues input vectors and aoutput vectors, 
  /// for example we can learn network with input:
  /// 1)Daytime
  /// 2)Day of the week
  /// 3)First char
  /// 4)Second char
  /// And output:
  /// 1)Time between first and second char
  /// 
  /// Or any other
  /// 
  /// Thats wright, when we learn neyron network we give input 
  /// and output vector of true person. But when we check we 
  /// Run network on the input vector and check output vector 
  /// with real output vector of the person and if they are alike 
  /// then we recognize the person.
  /// </summary>
  public class Network
  {
    #region private fields

    /// <summary>
    /// Name of the network. Used for creating samples for learning.
    /// </summary>
    private readonly ParameterName name;

    /// <summary>
    /// number of the input neyrons one of them will be output:
    /// firstChar,
    /// firstCode,
    /// firstCodePressTime,
    /// secondChar,
    /// secondCode,
    /// secondCodePressTime,
    /// milliseconds,
    /// hoursOfTheDay
    /// </summary>
    private int numberOfTheInputNeyrons;

    /// <summary>
    /// number of the output neyrons:
    /// 1)Time between first and second char
    /// </summary>
    private const int numberOfTheOutputNeyrons = 1;

    /// <summary>
    /// Represents a Backpropagation neural network.
    /// </summary>
    private NeuronDotNet.Core.Network network;

    #endregion

    #region private methods

    /// <summary>
    /// Log
    /// </summary>
    private static readonly CyclicTraceSource log =
  TraceSourceFactory.CreateEntity();

    /// <summary>
    /// Creates the network.
    /// </summary>
    /// <param name="learningRate">The learning rate.</param>
    /// <param name="neuronCount">The neuron count.</param>
    /// <param name="networkSchena">The network schena.</param>
    private void CreateNetwork(double learningRate, int neuronCount, IList<Layers> networkSchena)
    {
      //create layers
      LinearLayer inputLayer =
        new LinearLayer(numberOfTheInputNeyrons);

      ActivationLayer outputLayer = CreateNetwork(networkSchena,
        inputLayer, numberOfTheOutputNeyrons, neuronCount, learningRate);

      network = new BackpropagationNetwork(inputLayer, outputLayer);

      network.SetLearningRate(learningRate);
    }

    /// <summary>
    /// Creates the network by given schema.
    /// </summary>
    /// <param name="networkChema">The network chema.</param>
    /// <param name="inputLayer">The input layer.</param>
    /// <param name="numberOfTheOutputNeyrons">The number of the output neyrons.</param>
    /// <param name="neuronCount">The neuron count.</param>
    /// <param name="learningRate">The learning rate.</param>
    /// <returns>output layer</returns>
    private ActivationLayer CreateNetwork(
      IList<Layers> networkChema,
      ActivationLayer inputLayer,
      int numberOfTheOutputNeyrons,
      int neuronCount,
      double learningRate)
    {
      ActivationLayer layerToReturn = inputLayer;
      for (int i = 0; i < networkChema.Count; i++)
      {
        int numberOfNeyrons = i + 1 == networkChema.Count ?
          numberOfTheOutputNeyrons : neuronCount;

        switch (networkChema[i])
        {
          case Layers.LogarithmLayer:
            layerToReturn = CreateLayer(
              new LogarithmLayer(numberOfNeyrons),
              layerToReturn,
              learningRate);
            break;
          case Layers.SigmoidLayer:
            layerToReturn = CreateLayer(
              new SigmoidLayer(numberOfNeyrons),
              layerToReturn,
              learningRate);
            break;
          case Layers.SineLayer:
            layerToReturn = CreateLayer(
             new SineLayer(numberOfNeyrons),
             layerToReturn,
             learningRate);
            break;
          case Layers.TanhLayer:
            layerToReturn = CreateLayer(
             new TanhLayer(numberOfNeyrons),
             layerToReturn,
             learningRate);
            break;
        }
      }
      return layerToReturn;
    }

    /// <summary>
    /// Creates the layer.
    /// </summary>
    /// <param name="hiddenLayer">The hidden layer.</param>
    /// <param name="layerToReturn">The layer to return.</param>
    /// <param name="learningRate">The learning rate.</param>
    /// <returns></returns>
    private ActivationLayer CreateLayer(
      ActivationLayer hiddenLayer,
      ActivationLayer layerToReturn,
      double learningRate)
    {
      new BackpropagationConnector(layerToReturn, hiddenLayer).
        Initializer = new RandomFunction(0d, learningRate);

      layerToReturn = hiddenLayer;
      return layerToReturn;
    }

    /// <summary>
    /// Gets the results.
    /// </summary>
    /// <param name="trainingSet">The training set.</param>
    /// <returns></returns>
    private float GetResults(TrainingSet trainingSet)
    {
      double accuracy = 0.99;

      List<bool> results = new List<bool>();
      foreach (var sample in trainingSet.TrainingSamples)
      {
        double result = network.Run(sample.InputVector)[0];

        double waitingOuput = sample.OutputVector[0];

        if (waitingOuput != 0 && result != 0)
        {
          double tochnost = result / waitingOuput;
          results.Add(tochnost > accuracy && tochnost <= 1 ||
                   1 / tochnost > accuracy && 1 / tochnost <= 1);
        }
        else
        {
          results.Add(1 - accuracy > Math.Abs(result) && waitingOuput == 0 ||
                   1 - accuracy > Math.Abs(waitingOuput) && result == 0);
        }
      }
      float chislitel = 1;
      foreach (bool result in results)
      {
        if (result)
        {
          chislitel++;
        }
      }
      return chislitel / results.Count;
    }
    #endregion

    #region Constructors

    /// <summary>
    /// Initializes a new instance of the <see cref="Network"/> class.
    /// </summary>
    /// <param name="name">The name.</param>
    private Network(ParameterName name)
    {
      this.name = name;
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="Network"/> class.
    /// </summary>
    /// <param name="fileName">Name of the file with Network.</param>
    public Network(ParameterName name, String fileName)
      : this(name, new FileStream(fileName, FileMode.Open))
    {
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="Network"/> class.
    /// </summary>
    /// <param name="streamWithNetwork">The stream with network.</param>
    public Network(ParameterName name, Stream streamWithNetwork)
      : this(name)
    {
      // Construct a BinaryFormatter and use it 
      // to serialize the data to the stream.
      BinaryFormatter formatter = new BinaryFormatter();

      // Deserialize the array elements.
      streamWithNetwork.Position = 0;

      network = (BackpropagationNetwork)formatter.
        Deserialize(streamWithNetwork);

      streamWithNetwork.Dispose();
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="Network"/> class.
    /// </summary>
    /// <param name="numberOfTheInputNeyrons">The number of the input neyrons.</param>
    /// <param name="learningRate">the learning rate. All layers in
    /// the network will use this constant
    /// value as learning rate during the learning process.</param>
    /// <param name="neuronCount">Number of the Neyrons in the Hidden Layer.</param>
    /// <param name="schema">The schema.</param>
    public Network(ParameterName name, int numberOfTheInputNeyrons, double learningRate,
      int neuronCount, IList<Layers> schema)
      : this(name)
    {
      if (learningRate < 0)
      { learningRate = 0.25d; }
      if (neuronCount <= 0) { neuronCount = 4; }
      this.numberOfTheInputNeyrons = numberOfTheInputNeyrons;
      CreateNetwork(learningRate, neuronCount, schema);
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="Network"/> class.
    /// </summary>
    /// <param name="network">The network.</param>
    public Network(ParameterName name, NeuronDotNet.Core.Network network)
      : this(name)
    {
      this.network = network;
    }

    #endregion

    #region public methods

    /// <summary>
    /// True if neyron network finished the learning 
    /// process and ready to work.
    /// </summary>
    /// <value><c>true</c> if learned; otherwise, <c>false</c>.</value>
    public bool Learned
    {
      get;
      private set;
    }

    /// <summary>
    /// Learns the neyron network with specified trayning dictionary.
    /// </summary>
    /// <param name="trainingSamples">The training samples.</param>
    /// <param name="cycles">Number of training epochs.
    /// (All samples are trained in some random order, in every
    /// training epoch).</param>
    public float Learn(
      IEnumerable<UznavaykoSample> trainingSamples,
      int cycles, int outputDelimiter)
    {
      float resultToReturn = 0;

      if (cycles <= 0) { cycles = 10000; }
      if (trainingSamples == null)
      {
        return -1;
      }
      TrainingSet trainingSet = GetTrainingSet(trainingSamples, outputDelimiter);
      if (trainingSet == null)
      {
        return -1;
      }

      resultToReturn = IndividualSampleLearn(trainingSet);

      int i = 0;
      int numberOfDuplicates = 0;
      float maxRexult = resultToReturn;
      while (resultToReturn < 0.75 &&
            (i < 50 || resultToReturn < maxRexult) &&
             numberOfDuplicates < 5)
      {
        network.Learn(trainingSet, cycles);
        resultToReturn = GetResults(trainingSet);
        if (maxRexult == resultToReturn)
        {
          numberOfDuplicates++;
        }
        else
        {
          if (maxRexult < resultToReturn)
          {
            maxRexult = resultToReturn;
            numberOfDuplicates = 0;
          }
        }
        i++;
      }
      return maxRexult;
    }

    /// <summary>
    /// Gets the training set.
    /// </summary>
    /// <param name="trainingSamples">The training samples.</param>
    /// <param name="outputDelimiter">The output delimiter.</param>
    /// <returns></returns>
    public static TrainingSet GetTrainingSet(IEnumerable<UznavaykoSample> trainingSamples,
      int outputDelimiter)
    {
      TrainingSet trainingSet = null;
      Boolean setInitialized = false;

      UznavaykoSample oldSample = null;

      foreach (UznavaykoSample sample in
        trainingSamples)
      {
        if (!setInitialized)
        {
          oldSample = sample;
          trainingSet = new TrainingSet(7, 1);
          setInitialized = true;
          continue;
        }

        if (oldSample.nameOfTheOutput != sample.nameOfTheOutput)
        {
          throw new ArgumentException();
        }

        IDictionary<ParameterName, double> oldSampleDictionary = UznavaykoSample.
          ToDictionary(oldSample.nameOfTheOutput, oldSample);
        double secondCharOfFirstSample = oldSampleDictionary[ParameterName.secondChar];
        double secondPressTimeOfFirstSample = oldSampleDictionary[ParameterName.secondCodePressTime];

        IDictionary<ParameterName, double> sampleDictionary = UznavaykoSample.
         ToDictionary(sample.nameOfTheOutput, sample);
        double firstCharOfSecondSample = sampleDictionary[ParameterName.firstChar];
        double firstPressTimeOfSecondSample = sampleDictionary[ParameterName.firstCodePressTime];

        if (secondCharOfFirstSample == firstCharOfSecondSample &&
          secondPressTimeOfFirstSample == firstPressTimeOfSecondSample)
        {
          TrainingSample trainingSample = GetTrainingSample(oldSample.nameOfTheOutput,
            oldSampleDictionary,
            sampleDictionary, outputDelimiter);

          trainingSet.Add(trainingSample);
        }

        oldSample = sample;
      }
      return trainingSet;
    }

    /// <summary>
    /// Gets the training sample.
    /// </summary>
    /// <param name="nameOfTheOutput">The name of the output.</param>
    /// <param name="oldSample">The old sample.</param>
    /// <param name="sample">The sample.</param>
    /// <param name="outputDelimiter">The output delimiter.</param>
    /// <returns></returns>
    public static TrainingSample GetTrainingSample(
      ParameterName nameOfTheOutput,
      IDictionary<ParameterName, double> oldSample,
      IDictionary<ParameterName, double> sample, int outputDelimiter)
    {
      #region validation
      if (oldSample[ParameterName.secondCodePressTime] != sample[ParameterName.firstCodePressTime])
      {
        throw new InvalidDataException();
      }

      if (oldSample[ParameterName.secondChar] != sample[ParameterName.firstChar])
      {
        throw new InvalidDataException();
      }
      #endregion

      double pressTimeSum = oldSample[ParameterName.firstCodePressTime] +
                            oldSample[ParameterName.secondCodePressTime] +
                            sample[ParameterName.secondCodePressTime];

      double millisecondsSum = sample[ParameterName.milliseconds] +
                            oldSample[ParameterName.milliseconds];


      List<double> inputVector = new List<double>();
      List<double> outputVector = new List<double>();

      //this parameters are default for the input vector
      inputVector.Add(oldSample[ParameterName.firstCodePressTime] / pressTimeSum);
      inputVector.Add(oldSample[ParameterName.milliseconds] / millisecondsSum);
      inputVector.Add(oldSample[ParameterName.firstChar] / outputDelimiter);

      //firstChar
      if (nameOfTheOutput == ParameterName.firstChar)
      { outputVector.Add(sample[ParameterName.firstChar] / outputDelimiter); }
      else
      { inputVector.Add(sample[ParameterName.firstChar] / outputDelimiter); }
      //secondChar
      if (nameOfTheOutput == ParameterName.secondChar)
      { outputVector.Add(sample[ParameterName.secondChar] / outputDelimiter); }
      else
      { inputVector.Add(sample[ParameterName.secondChar] / outputDelimiter); }
      //firstCodePressTime
      if (nameOfTheOutput == ParameterName.firstCodePressTime)
      { outputVector.Add(sample[ParameterName.firstCodePressTime] / pressTimeSum); }
      else
      { inputVector.Add(sample[ParameterName.firstCodePressTime] / pressTimeSum); }
      //secondCodePressTime
      if (nameOfTheOutput == ParameterName.secondCodePressTime)
      { outputVector.Add(sample[ParameterName.secondCodePressTime] / pressTimeSum); }
      else
      { inputVector.Add(sample[ParameterName.secondCodePressTime] / pressTimeSum); }
      //milliseconds
      if (nameOfTheOutput == ParameterName.milliseconds)
      { outputVector.Add(sample[ParameterName.milliseconds] / millisecondsSum); }
      else
      { inputVector.Add(sample[ParameterName.milliseconds] / millisecondsSum); }

      return new TrainingSample(inputVector.ToArray(), outputVector.ToArray());
    }

    /// <summary>
    /// Individuals the sample learn.
    /// </summary>
    /// <param name="trainingSet">The training set.</param>
    /// <returns></returns>
    public float IndividualSampleLearn(TrainingSet trainingSet)
    {
      foreach (var sample in trainingSet.TrainingSamples)
      {
        int count = 0;
        int numberOfDuplicates = 0;
        double result = 0;
        const double accuracy = 0.99;
        bool needToLearn = false;
        double waitingOuput = sample.OutputVector[0];
        while (!needToLearn && count < 1000 && numberOfDuplicates < 5)
        {
          network.Learn(sample, 1, 2);
          double innerResult = network.Run(sample.InputVector)[0];

          if (result == innerResult)
          {
            numberOfDuplicates++;
          }
          else
          {
            result = innerResult;
          }

          double tochnost = result / waitingOuput;

          if (waitingOuput != 0 && result != 0)
          {

            needToLearn = tochnost > accuracy && tochnost <= 1 ||
                     1 / tochnost > accuracy && 1 / tochnost <= 1;
          }
          else
          {
            needToLearn = 1 - accuracy > Math.Abs(result) && waitingOuput == 0 ||
                     1 - accuracy > Math.Abs(waitingOuput) && result == 0;
          }

          count++;
        }
      }

      return GetResults(trainingSet);
    }

    /// <summary>
    /// Stops the learning of the Neyron Network.
    /// </summary>
    public void StopLearning()
    {
      Learned = true;
      network.StopLearning();
    }

    /// <summary>
    /// Saves network to stream.
    /// </summary>
    /// <param name="streamToSave">The straem to save.</param>
    public void SaveToStream(Stream streamToSave)
    {
      // Construct a BinaryFormatter and use it 
      // to serialize the data to the stream.
      BinaryFormatter formatter = new BinaryFormatter();

      // Serialize the array elements.
      formatter.Serialize(streamToSave, network);
    }

    /// <summary>
    /// Saves network to stream.
    /// </summary>
    /// <param name="fileToSave">The file to save.</param>
    public void SaveToFile(String fileToSave)
    {
      FileStream fileStream = new FileStream(fileToSave,
        FileMode.OpenOrCreate);

      // Construct a BinaryFormatter and use it 
      // to serialize the data to the stream.
      BinaryFormatter formatter = new BinaryFormatter();

      // Serialize the array elements.
      formatter.Serialize(fileStream, network);

      fileStream.Close();
    }

    /// <summary>
    /// Runs the neural network against the given input
    /// </summary>
    /// <param name="input">
    /// Input to the network
    /// </param>
    /// <returns>
    /// The output of the network
    /// </returns>
    /// <exception cref="ArgumentNullException">
    /// If <c>input</c> is <c>null</c>
    /// </exception>
    public double[] Run(double[] input)
    {
      return network.Run(input);
    }

    /// <summary>
    /// Learns the specified input.
    /// </summary>
    /// <param name="input">The input vector.</param>
    /// <param name="output">The output vector.</param>
    public void Learn(TrainingSample sample)
    {
      network.Learn(sample, 1, 100);
    }

    #endregion
  }
}
