﻿using System;
using System.Linq;
using System.IO;
using System.Collections.Generic;
using System.Threading;
using NeuronDotNet.Core;
using NeuronDotNet.Core.Backpropagation;
using NeuronDotNet.Core.Initializers;

namespace Uznavayko.NeyronNetwork.UznavaykoNetwork
{
  /// <summary>
  /// This is the class for neyron network for the Uznavayko project.
  /// 
  /// Uznavaykos neyron network inputs are:
  /// 1)Daytime
  /// 2)Day of the week
  /// 3)First char
  /// 4)Second char
  /// 5)Time between first and second char
  /// 
  /// we can give varioues input vectors and aoutput vectors, 
  /// for example we can learn network with input:
  /// 1)Daytime
  /// 2)Day of the week
  /// 3)First char
  /// 4)Second char
  /// And output:
  /// 1)Time between first and second char
  /// 
  /// Or any other
  /// 
  /// Thats wright, when we learn neyron network we give input 
  /// and output vector of true person. But when we check we 
  /// Run network on the input vector and check output vector 
  /// with real output vector of the person and if they are alike 
  /// then we recognize the person.
  /// </summary>
  public class ComplexNetwork
  {
    #region privat fields

    /// <summary>
    /// outputDelimiter
    /// </summary>
    private int outputDelimiter;

    /// <summary>
    /// the learning rate. All layers in 
    /// the network will use this constant
    /// value as learning rate during the learning process.
    /// </summary>
    private const double learningRate = 0.3d;

    /// <summary>
    /// numberOfTheInputNeyrons
    /// </summary>
    private int numberOfTheInputNeyrons;

    /// <summary>
    /// paramsThatDoNotCount
    /// </summary>
    private IEnumerable<ParameterName> paramsThatDoNotCount;

    /// <summary>
    /// Number of the Neurons in the Hidden Layer.
    /// </summary>
    private const int neyronCount = 10;

    /// <summary>
    /// Number of training epochs. 
    /// (All samples are trained in some random order, in every
    /// training epoch).
    /// </summary>
    private const int cycles = 1000;

    #endregion

    #region private methods

    /// <summary>
    /// Gets the input vector schema.
    /// </summary>
    /// <param name="paramsThatDoNotCount">The params that do not count.</param>
    /// <param name="outputVector">The output vector.</param>
    /// <returns></returns>
    private IEnumerable<ParameterName> GetInputVectorSchema(
      IEnumerable<ParameterName> paramsThatDoNotCount,
      ParameterName outputVector)
    {
      return from param in
               Enum.GetValues(typeof(ParameterName)).OfType<ParameterName>()
             where param != outputVector && !paramsThatDoNotCount.Contains(param)
             select param;
    }

    /// <summary>
    /// Sets the name of the neuron file.
    /// </summary>
    /// <param name="name">The name.</param>
    /// <param name="value">The value.</param>
    private void SetNeuronFileName(ParameterName name, string value)
    {
      if (neuronFileNames.Keys.Contains(name))
      {
        neuronFileNames[name] = value;
      }
      else
      {
        neuronFileNames.Add(name, value);
      }
    }

    #endregion

    #region public properties

    /// <summary>
    /// Contains Networks depending on output predicted parameter.
    /// </summary>
    public readonly Dictionary<ParameterName, Network>
      networksHolder =
      new Dictionary<ParameterName, Network>();


    /// <summary>
    /// Contains Networks depending on output predicted parameter.
    /// </summary>
    public readonly Dictionary<ParameterName, string>
      neuronFileNames =
      new Dictionary<ParameterName, string>();

    /// <summary>
    /// NetworkSchemesDictionary
    /// </summary>
    public readonly Dictionary<ParameterName, IList<Layers>> NetworkSchemesDictionary =
      new Dictionary<ParameterName, IList<Layers>>
        {
          {ParameterName.firstChar,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.firstCode,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.firstCodePressTime,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.milliseconds,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.secondChar,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.secondCode,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
          {ParameterName.secondCodePressTime,new List<Layers>{Layers.SigmoidLayer,Layers.SigmoidLayer}},
        };

    /// <summary>
    /// If true then we learn exceptions and do not tell about the errors.
    /// </summary>
    public bool LearningMode { get; set; }

    /// <summary>
    /// File name with networks data for SecondChar.
    /// </summary>
    public string NeuronSecondCharFileName
    {
      get { return neuronFileNames[ParameterName.secondChar]; }
      set { SetNeuronFileName(ParameterName.secondChar, value); }
    }

    /// <summary>
    /// File name with networks data for SecondCode.
    /// </summary>
    public string NeuronSecondCodeFileName
    {
      get { return neuronFileNames[ParameterName.secondCode]; }
      set { SetNeuronFileName(ParameterName.secondCode, value); }
    }

    /// <summary>
    /// File name with networks data for Second 
    /// chars key press time.
    /// </summary>
    public string NeuronSecondTimeFileName
    {
      get { return neuronFileNames[ParameterName.secondCodePressTime]; }
      set { SetNeuronFileName(ParameterName.secondCodePressTime, value); }
    }

    /// <summary>
    /// File name with networks data for milliseconds time.
    /// </summary>
    public string NeuronMilliseconds
    {
      get { return neuronFileNames[ParameterName.milliseconds]; }
      set { SetNeuronFileName(ParameterName.milliseconds, value); }
    }

    /// <summary>
    /// File name with networks data for FirstChar.
    /// </summary>
    public string NeuronFirstCharFileName
    {
      get { return neuronFileNames[ParameterName.firstChar]; }
      set { SetNeuronFileName(ParameterName.firstChar, value); }
    }

    /// <summary>
    /// File name with networks data for FirstCode.
    /// </summary>
    public string NeuronFirstCodeFileName
    {
      get { return neuronFileNames[ParameterName.firstCode]; }
      set { SetNeuronFileName(ParameterName.firstCode, value); }
    }

    /// <summary>
    /// File name with networks data for First 
    /// chars key press time.
    /// </summary>
    public string NeuronFirstTimeFileName
    {
      get { return neuronFileNames[ParameterName.firstCodePressTime]; }
      set { SetNeuronFileName(ParameterName.firstCodePressTime, value); }
    }

    #endregion

    #region public methods

    /// <summary>
    /// Learns the networks.
    /// </summary>
    /// <param name="samples">The samples.</param>
    public void LearnNetworks(
      Dictionary<ParameterName, IEnumerable<UznavaykoSample>> samples)
    {
      foreach (ParameterName key in networksHolder.Keys)
      {
        UznavaykoSample.InputVectorSchema = GetInputVectorSchema(paramsThatDoNotCount, key);

        if (!networksHolder[key].Learned)
        {
          networksHolder[key].Learn(
            samples[key], cycles, outputDelimiter);
        }
      }
    }

    /// <summary>
    /// Creates the networks. First try to load networks from
    /// UznavaykoFirstChar.dat and UznavaykoSecondChar.dat if they
    /// exists if no then create networks with default settings.
    /// if samples in collection are absent then using dataCollector
    /// collect data. If created networks are not learned then learn them
    /// using collections of samples.
    /// </summary>
    public void CreateNetwork(IEnumerable<ParameterName> paramsThatDoNotCount)
    {
      this.paramsThatDoNotCount = paramsThatDoNotCount;
      numberOfTheInputNeyrons = 7;
      if (String.IsNullOrEmpty(NeuronFirstCharFileName) ||
        String.IsNullOrEmpty(NeuronFirstCodeFileName) ||
        String.IsNullOrEmpty(NeuronFirstTimeFileName) ||
        String.IsNullOrEmpty(NeuronMilliseconds) ||
        String.IsNullOrEmpty(NeuronSecondCharFileName) ||
        String.IsNullOrEmpty(NeuronSecondCodeFileName) ||
        String.IsNullOrEmpty(NeuronSecondTimeFileName))
      {
        throw new ArgumentException("One of the names for network data files is empty.");
      }

      //CreateNetwork(ParameterName.firstChar, NeuronFirstCharFileName);

      //CreateNetwork(ParameterName.firstCode, NeuronFirstCodeFileName);

      CreateNetwork(ParameterName.firstCodePressTime, NeuronFirstTimeFileName);

      //CreateNetwork(ParameterName.milliseconds, NeuronMilliseconds);

      //CreateNetwork(ParameterName.secondChar, NeuronSecondCharFileName);

      //CreateNetwork(ParameterName.secondCode, NeuronSecondCodeFileName);

      CreateNetwork(ParameterName.secondCodePressTime, NeuronSecondTimeFileName);
    }

    /// <summary>
    /// Creates the networks. First try to load network from
    /// Uznavayko file if they
    /// exists if no then create network with default settings.
    /// </summary>
    /// <param name="name">The name.</param>
    /// <param name="fileName">Name of the file.</param>
    private void CreateNetwork(ParameterName name, string fileName)
    {
      if (File.Exists(fileName))
      {
        networksHolder.Add(name, new Network(name, fileName));
      }
      else
      {
        networksHolder.Add(
          name,
          new Network(name, numberOfTheInputNeyrons, learningRate, neyronCount,
            NetworkSchemesDictionary[name]));
      }
    }

    /// <summary>
    /// Gets the char result.
    /// </summary>
    /// <param name="parameterName">Name of the parameter.</param>
    /// <param name="sampleOld">The sample old.</param>
    /// <param name="sampleNew">The sample new.</param>
    /// <param name="accuracy">The accuracy.</param>
    /// <returns></returns>
    public bool GetCharResult(ParameterName parameterName,
     UznavaykoSample sampleOld, UznavaykoSample sampleNew, double accuracy)
    {
      if (!networksHolder.Keys.Contains(parameterName))
      {
        throw new ArgumentOutOfRangeException("parameterName");
      }

      IDictionary<ParameterName, double> oldSampleDictionary =
        UznavaykoSample.ToDictionary(sampleOld.nameOfTheOutput, sampleOld);

      IDictionary<ParameterName, double> newSampleDictionary =
        UznavaykoSample.ToDictionary(sampleNew.nameOfTheOutput, sampleNew);

      TrainingSample sample = Network.GetTrainingSample(parameterName,
        oldSampleDictionary,
        newSampleDictionary, outputDelimiter);

      double[] charOutputArray =
        networksHolder[parameterName].Run(sample.InputVector);

      double waitingOuput = sample.OutputVector[0];

      bool result;

      if (waitingOuput != 0 && charOutputArray[0] != 0)
      {
        double tochnost = charOutputArray[0] / waitingOuput;
        result = tochnost > accuracy && tochnost <= 1 ||
                 1 / tochnost > accuracy && 1 / tochnost <= 1;
      }
      else
      {
        result = 1 - accuracy > Math.Abs(charOutputArray[0]) && waitingOuput == 0 ||
                 1 - accuracy > Math.Abs(waitingOuput) && charOutputArray[0] == 0;
      }
      if (LearningMode)
      {
        if (!result)
        {
          networksHolder[parameterName].Learn(sample);
        }
      }
      return result;
    }

    /// <summary>
    /// Saves this instance to dat file.
    /// </summary>
    public void Save()
    {
      foreach (ParameterName name in Enum.GetValues(typeof(ParameterName)))
      {
        if (networksHolder.Keys.Contains(name))
        {
          networksHolder[name].SaveToFile(neuronFileNames[name]);
        }
      }
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="ComplexNetwork"/> class.
    /// </summary>
    /// <param name="delimiter">The delimiter.</param>
    public ComplexNetwork(int delimiter)
    {
      outputDelimiter = delimiter;
    }

    #endregion
  }
}
