using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;

namespace MouseAnalyzer.Wrapper
{
    public delegate void GlobalMouseEventHandler(object sender, GlobalMouseEventArgs e);

    class MouseHooksWrapper : IDisposable
    {

        private delegate void ManagedEventHandler(int code, UIntPtr wparam, IntPtr lparam);

        public event GlobalMouseEventHandler MouseAction;

        private int _threadId;

        private bool _isHookAlreadySet;

        private ManagedEventHandler _callbackFunction;

        [DllImport("WindowsHooks.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, EntryPoint = "SetManagedCallbackFunction")]
        private static extern void SetManagedCallbackFunction(ManagedEventHandler callbackFunc);

        [DllImport("WindowsHooks.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, EntryPoint = "SetMouseLLHook")]
        private static extern bool SetMouseLLHook(int threadID);

        [DllImport("WindowsHooks.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, EntryPoint = "RemoveMouseLLHook")]
        private static extern void RemoveMouseLLHook();

        [DllImport("WindowsHooks.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, EntryPoint = "GetCoordinatesFromLPARAM")]
        private static extern bool GetCoordinatesFromLPARAM(IntPtr lParam, ref int x, ref int y);

        public MouseHooksWrapper(int threadID)
        {
            _threadId = threadID;
            _isHookAlreadySet = false;
            _callbackFunction = new ManagedEventHandler(GenerateNewGlobalMouseEvent);
        }

        public MouseHooksWrapper() : this(0)
        {
        }

        ~MouseHooksWrapper()
        {
            Dispose();
        }

        public void SetGlobalMouseHook()
        {
            if (!_isHookAlreadySet)
            {
                SetManagedCallbackFunction(_callbackFunction);
                SetMouseLLHook(_threadId);
                _isHookAlreadySet = true;
            }
        }

        public void RemoveGlobalMouseHook()
        {
            if (_isHookAlreadySet)
            {
                RemoveMouseLLHook();
                _isHookAlreadySet = false;
            }
        }

        private void GenerateNewGlobalMouseEvent(int code, UIntPtr w, IntPtr l)
        {
            int x = 0;
            int y = 0;
            int mouseEventType = (int)w.ToUInt32();
            if (Enum.IsDefined(typeof(MouseEventType), mouseEventType) && GetCoordinatesFromLPARAM(l, ref x, ref y))
            {
                GlobalMouseEventArgs e = new GlobalMouseEventArgs((MouseEventType)mouseEventType, x, y);
                OnNewGlobalMouseAction(this, e);
            }
            else
            {
                GlobalMouseEventArgs e = new GlobalMouseEventArgs(mouseEventType, 0, 0);
                OnNewGlobalMouseAction(this, e);
            }
        }

        private void OnNewGlobalMouseAction(object sender, GlobalMouseEventArgs e)
        {
            GlobalMouseEventHandler tempEventHandler = MouseAction;
            if (tempEventHandler != null)
            {
                tempEventHandler(sender, e);
            }
        }

        #region IDisposable Members

        public void Dispose()
        {
            RemoveGlobalMouseHook();
        }

        #endregion
    }
}
