﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Windows.Forms;
using System.Xml.Serialization;
using HarryBonesSolutions.Performance;
using Utils.UserActivityMonitor;
using Uznavayko.NeyronNetwork.UznavaykoNetwork;
using UznavaykoNetwork;
using XmlSerialization;

namespace Uznavayko.DataCollector
{
  ///<summary>
  ///Main class for DataCollector. Collects data for neuron network.
  ///</summary>
  public class DataCollector //: Form
  {
    #region private class

    private class Data
    {
      public const string Note = "Note";
      public const string Key = "Key";
      public const string Id = "Id";
      public const string Timer = "Timer";
      public const string DateTime = "DateTime";
    }

    #endregion

    #region private props

    /// <summary>
    /// current note, created after user keyDown on keyboard can be
    /// changes in KeyPress.
    /// </summary>
    private Note note;

    /// <summary>
    /// Note that was sent to user previously.
    /// </summary>
    private Note oldNote;

    /// <summary>
    /// 
    /// </summary>
    private XmlSerialization.XmlSerialization xmlFile;

    /// <summary>
    /// This timer needed to count time between key pressings.
    /// </summary>
    private readonly ExecutionTimer timer = new ExecutionTimer();

    /// <summary>
    /// Local table for holding all data for working with
    /// notes.
    /// </summary>
    private readonly DataTable noteTable = new DataTable();

    #endregion

    #region private methods

    #region key event handlers
    /// <summary>
    /// Collectors the key pressed.
    /// </summary>
    /// <param name="sender">The sender.</param>
    /// <param name="e">The <see cref="System.EventArgs"/> 
    /// instance containing the event data.</param>
    private void CollectorKeyPressed(object sender, KeyPressEventArgs e)
    {
      note.charCode = e.KeyChar;
    }

    /// <summary>
    /// Collectors the key down event handler.
    /// </summary>
    /// <param name="sender">The sender.</param>
    /// <param name="e">The <see cref="System.Windows.Forms.KeyEventArgs"/> instance containing the event data.</param>
    private void CollectorKeyDown(object sender, KeyEventArgs e)
    {
      DataRow row = noteTable.NewRow();
      timer.Stop();

      int key = e.KeyValue;



      long oldId = note == null ? 0 : note.Id;
      note =
        new Note(timer.GetExecutionTime().Milliseconds, key)
          {
            Id = ++oldId
          };

      timer.Start();

      ExecutionTimer timerOfKeyDown = new ExecutionTimer();
      timerOfKeyDown.Start();

      row[Data.Id] = note.Id;
      row[Data.Note] = note;
      row[Data.Key] = key;
      row[Data.Timer] = timerOfKeyDown;
      row[Data.DateTime] = DateTime.Now;
      noteTable.Rows.Add(row);
    }

    /// <summary>
    /// Collectors the key up event handler.
    /// </summary>
    /// <param name="sender">The sender.</param>
    /// <param name="e">The <see cref="System.Windows.Forms.KeyEventArgs"/> instance containing the event data.</param>
    private void CollectorKeyUp(object sender, KeyEventArgs e)
    {
      //get key
      int key = e.KeyValue;
      //try to get row with key
      DataRow[] currentRows = noteTable.Select(Data.Key + "=" + key);
      if(currentRows.Length == 0)
      {
        return;
      }
      DataRow currentRow = currentRows[0];

      //erase the key because it can be duplicated after user press the same button
      currentRow[Data.Key] = int.MinValue;

      ExecutionTimer currentTimer = (ExecutionTimer)currentRow[Data.Timer];
      //stop the timer
      currentTimer.Stop();

      int milliseconds = currentTimer.GetExecutionTime().Milliseconds;

      //this is needed in case we get 0
      milliseconds++;

      //disposing useless timer
      currentTimer.Dispose();

      //removing Timer from data table
      currentRow[Data.Timer] = null;

      Note note = (Note)currentRow[Data.Note];

      note.pressedTime = milliseconds;

      //if we have not previous 2 notes then we can't send it 
      //or create a word, because we can't check that user 
      //released the keys
      if (note.Id == 0)
      {
        return;
      }
      if (oldNote == null)
      {
        oldNote = note;
        return;
      }
      if (KeyPressed != null)
      {
        KeyPressed(CreateSender(note,
          ((DateTime)currentRow[Data.DateTime]).TimeOfDay.Hours),
          EventArgs.Empty);
      }
    }

    #endregion

    /// <summary>
    /// Gets the word array.
    /// </summary>
    /// <returns></returns>
    private Word[] GetWordArray()
    {
      List<Word> words = new List<Word>();
      int wordNumber = 0;

      List<DataRow> listOfNotes = noteTable.Select().ToList();
      for (int i = 1; i < listOfNotes.Count; )
      {
        words.Add(
          new Word(
                   wordNumber,
                   (DateTime)listOfNotes[i][Data.DateTime],
                   new[] { 
                     (Note)listOfNotes[i-1][Data.Note],
                     (Note)listOfNotes[i][Data.Note]}
                   )
                  );
        wordNumber++;
        i ++;
      }
      return words.ToArray();
    }

    /// <summary>
    /// Creates the sender to send in the event.
    /// </summary>
    /// <param name="notes">The notes.</param>
    /// <returns></returns>
    private Dictionary<ParameterName, double> CreateSender(Note notes, int hours)
    {
      Dictionary<ParameterName, double> dic =
        new Dictionary<ParameterName, double>
                  {
                    {ParameterName.firstChar, oldNote.charCode},
                    {ParameterName.firstCode, oldNote.characterCode},
                    {ParameterName.firstCodePressTime, oldNote.pressedTime},
                    {ParameterName.milliseconds,notes.miliseconds},
                    {ParameterName.secondChar, notes.charCode},
                    {ParameterName.secondCode, notes.characterCode},
                    {ParameterName.secondCodePressTime, notes.pressedTime},
                    {ParameterName.hoursOfTheDay, hours},
                  };
      oldNote = notes;
      return dic;
    }

    #endregion

    #region public events and methods

    /// <summary>
    /// Occurs when user key pressed.
    /// </summary>
    public event EventHandler KeyPressed;


    /// <summary>
    /// Saves the data.
    /// </summary>
    public void SaveData()
    {
      xmlFile.NoteList.Words = GetWordArray();
      xmlFile.WriteToFile();
    }

    /// <summary>
    /// Starts the collector.
    /// </summary>
    public void StartCollector()
    {
      HookManager.KeyPress += CollectorKeyPressed;
      HookManager.KeyDown += CollectorKeyDown;
      HookManager.KeyUp += CollectorKeyUp;
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="DataCollector"/> class.
    /// </summary>
    /// <param name="fileName">Name of the file.</param>
    public DataCollector(string fileName)
    {
      xmlFile = new XmlSerialization.XmlSerialization(fileName);
      noteTable.Columns.AddRange
       (
       new[]
          {
            new DataColumn(Data.Id,typeof(long)),
            new DataColumn(Data.Note,typeof(Note)),
            new DataColumn(Data.Key,typeof(int)),
            new DataColumn(Data.Timer,typeof(ExecutionTimer)),
            new DataColumn(Data.DateTime,typeof(DateTime))
          }
       );
    }
    #endregion
  }
}